-------------------------------------------------------------------------------
-- Elementarist 1.6
--
-- Shows the advised spell for an elemental shaman for optimal DPS output.
-------------------------------------------------------------------------------

Elementarist = {Locals = {}}

local L = Elementarist.Locals

Elementarist.versionNumber = '1.6'
Elementarist.playerName = UnitName("player")
Elementarist.playerGUID = UnitGUID("player")
Elementarist.targetGUID = nil
Elementarist.spellHaste = GetCombatRatingBonus(20)
Elementarist.timeSinceLastUpdate = 0
Elementarist.debuffTrackerUpdate = 0
Elementarist.spellPower = GetSpellBonusDamage(4);	-- nature spell bonus
Elementarist.lastBaseGCD = 1.5
Elementarist.person = {
	["foeCount"]	= 0,
	["friendCount"]	= 0,
	["friend"]  = {},
	["foe"]		= {}
}
Elementarist.lastPersonTablePurged = 0.0;
Elementarist.configPanel = nil
Elementarist.prevDB = {}
Elementarist.DPSTable = {}
Elementarist.DebugMode = false
Elementarist.inParty = 0
Elementarist.OmniCC = _G['OmniCC']
Elementarist.isEle = false
Elementarist.talentUnsure = true
Elementarist.lastSpell = nil
Elementarist.CustomIDs = {
	["Flask of the North Item"] = 47499,
	["Flask of the North Spell"] = 67016
}
Elementarist.SpellList = {
	["Flame Shock"] = GetSpellInfo(49233),
	["Lightning Bolt"] = GetSpellInfo(49238),
	["Lava Burst"] = GetSpellInfo(60043),
	["Chain Lightning"] = GetSpellInfo(49271),
	["Thunderstorm"] = GetSpellInfo(59159),
	["Purge"]	= GetSpellInfo(8012),
	["Wind Shear"] = GetSpellInfo(57994),
	["Water Shield"] = GetSpellInfo(57960),
	["Flametongue Weapon"] = GetSpellInfo(58790),
	["Totem of Wrath Totem"] = GetSpellInfo(57722),
	["Wrath of Air Totem"] = GetSpellInfo(3738),
	["Mana Spring Totem"] = GetSpellInfo(58777),
	["Totem of Wrath"] = GetSpellInfo(63283),
	["Elemental Mastery"] = GetSpellInfo(16166),
	["Fire Nova"] = GetSpellInfo(61657),
	
	-- racials
	["Berserking"] = GetSpellInfo(26297),	-- Troll racial
	["Blood Fury"] = GetSpellInfo(33697),	-- Orc racial
	
	-- debuffs
	["Totem of Wrath Debuff"] = GetSpellInfo(30708),
	["Heart of the Crusader"] = GetSpellInfo(54499),
	
	-- other buffs
	["Demonic Pact"] = GetSpellInfo(48090),
	["Flask of the North"] = GetSpellInfo(Elementarist.CustomIDs["Flask of the North Spell"]),
	["Flask of the Frost Wyrm"] = GetSpellInfo(53755),
}
Elementarist.textureList = {
	["next"] = nil,
	["next1"] = nil,
	["next2"] = nil,
	["Misc"] = nil,
	["int"] = nil,
	["debuff"] = nil
}

Elementarist.textList = {
	["dps"] = nil,
	["debuff"] = nil
}
Elementarist.Behaviours = {
}
Elementarist.CLBehaviours = {
}
Elementarist.HostileFilter = {
  ["_DAMAGE"] = true, 
  ["_LEECH"] = true,
  ["_DRAIN"] = true,
  ["_STOLEN"] = true,
  ["_INSTAKILL"] = true,
  ["_INTERRUPT"] = true,
  ["_MISSED"] = true
}

-- Our sneaky frame to watch for events ... checks Elementarist.events[] for the function.  Passes all args.
Elementarist.eventFrame = CreateFrame("Frame")
Elementarist.eventFrame:SetScript("OnEvent", function(this, event, ...)
  Elementarist.events[event](...)
end)

Elementarist.eventFrame:RegisterEvent("ADDON_LOADED")
Elementarist.eventFrame:RegisterEvent("PLAYER_LOGIN")
Elementarist.eventFrame:RegisterEvent("PLAYER_ALIVE")

-- Define our Event Handlers here
Elementarist.events = {}

function Elementarist:Debug(statictxt,msg)
	if (Elementarist.DebugMode) then
		if (msg) then
			DEFAULT_CHAT_FRAME:AddMessage("ELEDBG: ".. statictxt  .. " : " .. msg)
		else
			DEFAULT_CHAT_FRAME:AddMessage("ELEDBG: ".. statictxt  .. " : " .. "<nil>")
		end
	end
end

function Elementarist.events.PLAYER_TALENT_UPDATE()
	Elementarist:detectTalent()
	
	Elementarist:ApplySettings()
end

function Elementarist.events.PARTY_MEMBERS_CHANGED()
	Elementarist.inParty = Elementarist:PlayerInParty()
end

function Elementarist.events.PLAYER_ALIVE()
	-- check anything
	Elementarist:detectTalent()
	Elementarist:ApplySettings()
	
	-- Elementarist.eventFrame:UnregisterEvent("PLAYER_ALIVE")
end

function Elementarist.events.PLAYER_ENTERING_WORLD()
	Elementarist:detectTalent()
end

function Elementarist.events.PLAYER_LOGIN()
	Elementarist.playerName = UnitName("player");

	Elementarist.spellHaste = GetCombatRatingBonus(20)
	Elementarist.spellPower = GetSpellBonusDamage(4); 
end

function Elementarist.events.ADDON_LOADED(addon)
	if addon == "OmniCC" then
		Elementarist.OmniCC = _G['OmniCC']
	end
	if addon ~= "Elementarist" then return end
	local _,playerClass = UnitClass("player")
	if playerClass ~= "SHAMAN" then
		Elementarist.eventFrame:UnregisterEvent("PLAYER_ALIVE")
		return 
	end

	-- load defaults, if first start
	Elementarist:InitSettings()
	
	-- add slash command
	SlashCmdList["Elementarist"] = Elementarist.Options
	SLASH_Elementarist1 = "/Elementarist"
	SLASH_Elementarist2 = "/ele"
	
	-- check if talent is elemental
	Elementarist:detectTalent()

	Elementarist.playerLevel = UnitLevel("player")

	-- Setup behaviours
	Elementarist.Behaviours["1"] = L.BEHAVIOUR_KEEP_FS_UP
	Elementarist.Behaviours["2"] = L.BEHAVIOUR_FS_BEFORE_LVB
	Elementarist.CLBehaviours["0"] = L.CLSTBEHAVIOUR_NONE
	Elementarist.CLBehaviours["1"] = L.CLSTBEHAVIOUR_CL_AFTER_LVB
	Elementarist.CLBehaviours["2"] = L.CLSTBEHAVIOUR_CL_ON_CD
	
	-- Create GUI
	Elementarist:CreateGUI()
	Elementarist.displayFrame:SetScale(ElementaristDB.scale)
	Elementarist.OmniCC = _G['OmniCC']

	-- Create config page
	Elementarist:CreateConfig()
	
	-- Register for Function Events
	Elementarist.eventFrame:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
	Elementarist.eventFrame:RegisterEvent("COMBAT_RATING_UPDATE") -- Monitor the all-mighty haste
	Elementarist.eventFrame:RegisterEvent("PLAYER_TARGET_CHANGED")
	Elementarist.eventFrame:RegisterEvent("PLAYER_REGEN_ENABLED") -- Left combat, clean up all enemy GUIDs
	Elementarist.eventFrame:RegisterEvent("PLAYER_TALENT_UPDATE")
	Elementarist.eventFrame:RegisterEvent("PARTY_MEMBERS_CHANGED")
end

function Elementarist:InitSettings()
	if not ElementaristDB then 
		ElementaristDB = {} -- fresh start
	end
	if not ElementaristDB.scale then ElementaristDB.scale = 1 end
	if not ElementaristDB.debuffscale then ElementaristDB.debuffscale = 1 end
	if ElementaristDB.locked == nil then ElementaristDB.locked = false end
	if ElementaristDB.enabled == nil then ElementaristDB.enabled = true end
	if ElementaristDB.disableIfNotEle == nil then ElementaristDB.disableIfNotEle = true end
	if ElementaristDB.debuffdisabled == nil then ElementaristDB.debuffdisabled = false end
	if ElementaristDB.alpha == nil then ElementaristDB.alpha = 0.8 end
	if ElementaristDB.debuffalpha == nil then ElementaristDB.debuffalpha = 1 end
	if ElementaristDB.FireNova == nil then ElementaristDB.FireNova = true end
	if ElementaristDB.Behaviour == nil then ElementaristDB.Behaviour = Elementarist.Behaviours["1"] end
	if ElementaristDB.CLBehaviour == nil then ElementaristDB.CLBehaviour = Elementarist.CLBehaviours["0"] end
	if ElementaristDB.ThreatWarning == nil then ElementaristDB.ThreatWarning = true end
	if not ElementaristDB.x then ElementaristDB.x = -200 end
	if not ElementaristDB.y then ElementaristDB.y = -200 end
	if not ElementaristDB.relativePoint then ElementaristDB.relativePoint = "CENTER" end
	if not ElementaristDB.debuffx then ElementaristDB.debuffx = -200 end
	if not ElementaristDB.debuffy then ElementaristDB.debuffy = -100 end
	if not ElementaristDB.debuffrelativePoint then ElementaristDB.debuffrelativePoint = "CENTER" end
end

function Elementarist:detectTalent()
	local _,_,rest,_ = GetTalentTabInfo(3)
	local _,_,enha,_ = GetTalentTabInfo(2)
	local _,_,elem,_ = GetTalentTabInfo(1)

	if (elem+rest+enha>0) then
		Elementarist.isEle = ((elem>=enha) and (elem>=rest))
	end
	if (elem + enha + rest == 0) then
		Elementarist.talentUnsure = true
	else
		Elementarist.talentUnsure = false
	end
end

function Elementarist:PlayerInParty()
	if (GetNumRaidMembers()>0) then
		return 2
	elseif (GetNumPartyMembers()>0) then
		return 1
	else
		return 0
	end
end

function Elementarist:PurgePersonTable()
	for i,v in pairs(Elementarist.person["foe"]) do
		if (time()-v > 2) then
			-- no activity from that unit in last 2 seconds, remove it
			-- DEFAULT_CHAT_FRAME:AddMessage("Enemy removed")
			Elementarist.person["foe"][i] = 0	-- mark as inactive
			Elementarist.person["foeCount"] = Elementarist.person["foeCount"] - 1
		end
	end
	for i,v in pairs(Elementarist.person["friend"]) do
		if (GetTime()-v > 2) then
			-- no activity from that unit in last 2 seconds, remove it
			Elementarist.person["friend"][i] = 0	-- mark as inactive
			Elementarist.person["friendCount"] = Elementarist.person["friendCount"] - 1
		end
	end
	Elementarist.lastPersonTablePurged = GetTime()
end

function Elementarist:CountPerson(time, event, sguid, sname, sflags, dguid, dname, dflags)
	local suffix = event:match(".+(_.-)$")
	local stype = (tonumber(sguid:sub(5,5), 16)) % 8
	local dtype = (tonumber(dguid:sub(5,5), 16)) % 8
	if Elementarist.HostileFilter[suffix] then
		if (bit.band(sflags, COMBATLOG_OBJECT_AFFILIATION_MASK) < 8) and ((dtype==0) or (dtype==3)) then
			if ((not Elementarist.person["foe"][dguid]) or (Elementarist.person["foe"][dguid]==0)) then
				Elementarist.person["foeCount"] = Elementarist.person["foeCount"] + 1
			end
			Elementarist.person["foe"][dguid] = time
    	elseif (bit.band(dflags, COMBATLOG_OBJECT_AFFILIATION_MASK) < 8) and ((stype==0) or (stype==3)) then
			if ((not Elementarist.person["foe"][sguid]) or (Elementarist.person["foe"][sguid]==0)) then
				Elementarist.person["foeCount"] = Elementarist.person["foeCount"] + 1
			end
			Elementarist.person["foe"][sguid] = time
		end
		if (bit.band(sflags, COMBATLOG_OBJECT_AFFILIATION_MASK) >= 8) and ((dtype==0) or (dtype==3)) then
			if ((not Elementarist.person["friend"][dguid]) or (Elementarist.person["friend"][dguid]==0)) then
				Elementarist.person["friendCount"] = Elementarist.person["friendCount"] + 1
			end
			Elementarist.person["friend"][dguid] = time
    	elseif (bit.band(dflags, COMBATLOG_OBJECT_AFFILIATION_MASK) >= 8) and ((stype==0) or (stype==3)) then
			if ((not Elementarist.person["friend"][sguid]) or (Elementarist.person["friend"][sguid]==0)) then
				Elementarist.person["friendCount"] = Elementarist.person["friendCount"] + 1
			end
			Elementarist.person["friend"][sguid] = time
		end
	end
	if (Elementarist.lastPersonTablePurged < (GetTime() - 3)) and (Elementarist.person["foeCount"]>0) then
		Elementarist:PurgePersonTable()
	end
end

function Elementarist.HighDMGFormat(dmg_amount)
	if (dmg_amount >= 10000) then
		return(format('%.1f',dmg_amount/1000) .. "K")
	else
		return(format('%.f',dmg_amount))
	end
end

function Elementarist.events.COMBAT_LOG_EVENT_UNFILTERED(timestamp, event, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellId, spellName, spellSchool, damage, ...)
	if Elementarist.isEnabled() then
		if srcName == Elementarist.playerName then
			Elementarist:DecideSpells()
			-- calculate DPS
			if (event=="SPELL_CAST_SUCCESS") then
				Elementarist.lastSpell = spellName
			end
			if ((event=="SPELL_DAMAGE") or (event=="SPELL_PERIODIC_DAMAGE")) then
				-- DEFAULT_CHAT_FRAME:AddMessage("Enemy: "..dstGUID.. ", "..Elementarist.targetGUID)
				if (not Elementarist.DPSTable[dstGUID]) then
					Elementarist.DPSTable[dstGUID] = {
						["time"] = GetTime(),
						["amount"] = 0,
					}
				end
				
				Elementarist.DPSTable[dstGUID]["amount"] = Elementarist.DPSTable[dstGUID]["amount"] + damage
				local dps_txt = ""
				if (dstGUID == Elementarist.targetGUID) and (Elementarist.DPSTable[Elementarist.targetGUID]) then
					local dps_sec = GetTime() - Elementarist.DPSTable[dstGUID]["time"]
					if (dps_sec > 5) then
						dps_txt = format('%.f',(Elementarist.DPSTable[dstGUID]["amount"] / dps_sec))
					end
					local threat_txt = ""
					local _, status, threatpct, _, _ = UnitDetailedThreatSituation("player", "target")
					if (status) then
						if (threatpct<80) then
							threat_txt = format("%.f",threatpct) .. " %"
							Elementarist.cooldownFrame:SetReverse(false)
							if (ElementaristDB.ThreatWarning) and (Elementarist.inParty>0) and (threatpct>70) then
								RaidNotice_AddMessage(RaidBossEmoteFrame, L.THREAT_WARNING_PREFIX .. format("%.f",threatpct) .. L.THREAT_WARNING_SUFFIX, ChatTypeInfo["RAID_WARNING"])
							end
						else
							threat_txt = "|cffff0000" .. format("%.f",threatpct) .. " %|r"
							if (ElementaristDB.ThreatWarning) and (Elementarist.inParty>0) then
								RaidNotice_AddMessage(RaidBossEmoteFrame, "|cffff0000" .. L.THREAT_WARNING_PREFIX .. format("%.f",threatpct) ..  L.THREAT_WARNING_SUFFIX .. "|r", ChatTypeInfo["RAID_WARNING"])
							end
							Elementarist.cooldownFrame:SetReverse((Elementarist.person["friendCount"]>1) and (Elementarist.inParty>0))
						end
					end
					dps_txt = dps_txt .. "|n" .. threat_txt
					Elementarist.textList["dps"]:SetText(dps_txt)
				end
			end
		else
			-- count enemies if player in combat
			if (UnitAffectingCombat("player")) then
				-- enemy count for CL advise and multiple player in combat (for aggro warning)
				Elementarist:CountPerson(timestamp, event, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags)
			end
		end
	end
end

function Elementarist.events.COMBAT_RATING_UPDATE(unit)
	if unit == "player" then
    	Elementarist.spellHaste = GetCombatRatingBonus(20) -- update spell haste
		Elementarist.spellPower = GetSpellBonusDamage(4);	
	end
end

function Elementarist.events.PLAYER_TARGET_CHANGED(...) 
	Elementarist.targetGUID = UnitGUID("target")
	Elementarist.inParty = Elementarist:PlayerInParty()

	if (ElementaristDB.disableIfNotEle) and (not Elementarist:isEnabled()) then
		Elementarist:detectTalent()
		if Elementarist:isEnabled() then
			Elementarist:ApplySettings()
		end
	end
	if (not Elementarist.targetGUID) then
		local threat_txt = ""
		local _, status, threatpct, _, _ = UnitDetailedThreatSituation("player", "target")
		if (status) then
			if (threatpct<80) then
				threat_txt = format("%.f",threatpct) .. " %"
				Elementarist.cooldownFrame:SetReverse(false)
			else
				threat_txt = "|cffff0000" .. format("%.f",threatpct) .. " %|r"
				Elementarist.cooldownFrame:SetReverse((Elementarist.person["friendCount"]>1) and (Elementarist.inParty>0))
			end
		end
		Elementarist.textList["dps"]:SetText(threat_txt)
	end
	Elementarist:DecideSpells()
end

function Elementarist.events.PLAYER_REGEN_ENABLED(...)
	-- left combat
	Elementarist.person["friend"] = {}
	Elementarist.person["friendCount"] = 0
	Elementarist.person["foe"] = {}
	Elementarist.person["foeCount"] = 0
	Elementarist.DPSTable = {}
	Elementarist.textList["dps"]:SetText("")
	Elementarist.textList["debuff"]:SetText("")
	Elementarist.cooldownFrame:SetReverse(false)
	Elementarist:PurgePersonTable()
end

function Elementarist:isEnabled()
	if (Elementarist.talentUnsure) then
		Elementarist:detectTalent()
	end
	return (
		ElementaristDB.enabled and (
			(not ElementaristDB.disableIfNotEle) or (Elementarist.isEle)
		)
	)
end

function Elementarist:UpdateDebuffTracker()
	local name, _, icon, _, _, d, e = Elementarist:hasDeBuff("target",Elementarist.SpellList["Flame Shock"],"player")

	Elementarist.debuffTrackerUpdate = GetTime()
	if (name) then
		Elementarist.textureList["debuff"]:SetTexture(icon)
		if (not Elementarist.OmniCC) then
			Elementarist.textList["debuff"]:SetText(format('%.1f', (e - GetTime())))
		end
		Elementarist.debuffCooldownFrame:SetCooldown( e-d, d)
		
	else
		Elementarist.textureList["debuff"]:SetTexture(nil)
		Elementarist.textList["debuff"]:SetText("")
		Elementarist.debuffCooldownFrame:SetCooldown(0, 0)
	end
end

function Elementarist:ResetPosition()
	ElementaristDB.x = -200
	ElementaristDB.y = -200
	ElementaristDB.relativePoint = "CENTER"
	ElementaristDB.debuffx = -200
	ElementaristDB.debuffy = -100
	ElementaristDB.debuffrelativePoint = "CENTER"
	Elementarist.displayFrame:ClearAllPoints()
	Elementarist.displayFrame:SetPoint(ElementaristDB.relativePoint,ElementaristDB.x,ElementaristDB.y)
	Elementarist.debuffTracker:ClearAllPoints()
	Elementarist.debuffTracker:SetPoint(ElementaristDB.debuffrelativePoint,ElementaristDB.debuffx,ElementaristDB.debuffy)
end

function Elementarist:CreateGUI()

	local displayFrame = CreateFrame("Frame","ElementaristDisplayFrame",UIParent)
	displayFrame:SetFrameStrata("BACKGROUND")
	displayFrame:SetWidth(150)
	displayFrame:SetHeight(120)
	displayFrame:SetBackdrop({
          bgFile = "Interface\\Tooltips\\UI-Tooltip-Background", tile = true, tileSize = 32,
	})
	displayFrame:SetBackdropColor(0, 0, 0, .0)
	if (not ElementaristDB.locked) then
		displayFrame:SetBackdropColor(0, 0, 0, .3)
		displayFrame:EnableMouse(true)
		displayFrame:SetMovable(true)
		displayFrame:SetClampedToScreen(true)
		displayFrame:SetScript("OnMouseDown", function(self) self:StartMoving(); self:SetBackdropColor(0, 0, 0, .6); end)
		displayFrame:SetScript("OnMouseUp", function(self)
			self:StopMovingOrSizing()
			if (Elementarist.locked) then
				self:SetBackdropColor(0, 0, 0, 0)
			else
				self:SetBackdropColor(0, 0, 0, .3)
			end
			local _,_,rp,x,y = self:GetPoint()
			ElementaristDB.x = x
			ElementaristDB.y = y
			ElementaristDB.relativePoint = rp
		end)
		displayFrame:SetScript("OnDragStop", function(self)
			self:StopMovingOrSizing();
			if (Elementarist.locked) then
				self:SetBackdropColor(0, 0, 0, 0)
			else
				self:SetBackdropColor(0, 0, 0, .3)
			end
			local _,_,rp,x,y = self:GetPoint()
			ElementaristDB.x = x
			ElementaristDB.y = y
			ElementaristDB.relativePoint = rp
		end)
	end
	displayFrame:SetPoint(ElementaristDB.relativePoint,ElementaristDB.x,ElementaristDB.y)
	
	local displayFrame_next = CreateFrame("Frame","$parent_next", ElementaristDisplayFrame)
	local displayFrame_next1 = CreateFrame("Frame","$parent_next1", ElementaristDisplayFrame)
	local displayFrame_next2 = CreateFrame("Frame","$parent_next2", ElementaristDisplayFrame)
	local displayFrame_misc = CreateFrame("Frame","$parent_misc", ElementaristDisplayFrame)
	local displayFrame_int = CreateFrame("Frame","$parent_int", ElementaristDisplayFrame)
	local displayFrame_dps = CreateFrame("Frame","$parent_int", ElementaristDisplayFrame)
	
	displayFrame_next:SetWidth(60)
	displayFrame_next1:SetWidth(40)
	displayFrame_next2:SetWidth(20)
	displayFrame_misc:SetWidth(40)
	displayFrame_int:SetWidth(40)
	displayFrame_dps:SetWidth(60)

	displayFrame_next:SetFrameLevel(10)
	displayFrame_next1:SetFrameLevel(5)
	displayFrame_next2:SetFrameLevel(0)
	
	displayFrame_next:SetHeight(60)
	displayFrame_next1:SetHeight(40)
	displayFrame_next2:SetHeight(20)
	displayFrame_misc:SetHeight(40)
	displayFrame_int:SetHeight(40)
	displayFrame_dps:SetHeight(30)
	
	
	displayFrame_next:SetPoint("TOPLEFT", 45, -30)
	displayFrame_next1:SetPoint("TOPLEFT", 55, -10)
	displayFrame_next2:SetPoint("TOPLEFT", 65, 0)
	
	displayFrame_misc:SetPoint("TOPLEFT", 0, -80)
	displayFrame_int:SetPoint("TOPLEFT", 110, -80)
	displayFrame_dps:SetPoint("TOPLEFT", 45, -90)
	
	
	t = displayFrame_next:CreateTexture(nil,"BACKGROUND")
	t:SetTexture(nil)
	t:SetAllPoints(displayFrame_next)
	t:SetAlpha(1)
	displayFrame_next.texture = t
	Elementarist.textureList["next"] = t

	t = displayFrame_next1:CreateTexture(nil,"BACKGROUND")
	t:SetTexture(nil)
	t:SetAllPoints(displayFrame_next1)
	t:SetAlpha(0.7)
	displayFrame_next1.texture = t
	Elementarist.textureList["next1"] = t

	t = displayFrame_next2:CreateTexture(nil,"BACKGROUND")
	t:SetTexture(nil)
	t:SetAllPoints(displayFrame_next2)
	t:SetAlpha(0.5)
	displayFrame_next2.texture = t
	Elementarist.textureList["next2"] = t
	
	t = displayFrame_misc:CreateTexture(nil,"BACKGROUND")
	t:SetTexture(nil)
	t:SetAllPoints(displayFrame_misc)
	t:SetAlpha(1)
	displayFrame_misc.texture = t
	Elementarist.textureList["Misc"] = t
	
	t = displayFrame_int:CreateTexture(nil,"BACKGROUND")
	t:SetTexture(nil)
	t:SetAllPoints(displayFrame_int)
	t:SetAlpha(1)
	displayFrame_int.texture = t
	Elementarist.textureList["int"] = t
	
	t = displayFrame_dps:CreateFontString("$parentText","ARTWORK","GameFontNormal");
	t:SetAllPoints(displayFrame_dps)
	t:SetAlpha(1)
	t:SetText("")
	Elementarist.textList["dps"] = t


	displayFrame:SetScript("OnUpdate", function(this, elapsed)
		Elementarist:OnUpdate(elapsed)
	end)
  
	local cooldownFrame = CreateFrame("Cooldown","$parent_cooldown", ElementaristDisplayFrame_current, "CooldownFrameTemplate")
	cooldownFrame:SetHeight(60)
	cooldownFrame:SetWidth(60)
	cooldownFrame:ClearAllPoints()
	cooldownFrame:SetPoint("CENTER", displayFrame_next, "CENTER", 0, 0)
	
	displayFrame:SetAlpha(ElementaristDB.alpha)
	
	Elementarist.displayFrame = displayFrame
	Elementarist.displayFrame_next = displayFrame_next
	Elementarist.displayFrame_next1 = displayFrame_next1
	Elementarist.displayFrame_next2 = displayFrame_next2
	Elementarist.displayFrame_misc =  displayFrame_misc
	Elementarist.displayFrame_int =  displayFrame_int
	Elementarist.displayFrame_dps =  displayFrame_dps
	Elementarist.cooldownFrame = cooldownFrame
	
	local debuffTracker = CreateFrame("Frame","ElementaristDebuffTrackerFrame",UIParent)
	debuffTracker:SetFrameStrata("BACKGROUND")
	debuffTracker:SetWidth(50)
	debuffTracker:SetHeight(50)
	debuffTracker:SetBackdrop({
          bgFile = "Interface\\Tooltips\\UI-Tooltip-Background", tile = true, tileSize = 32,
	})
	debuffTracker:SetBackdropColor(0, 0, 0, .0)
	if (not ElementaristDB.locked) then
		debuffTracker:SetBackdropColor(0, 0, 0, .3)
		debuffTracker:EnableMouse(true)
		debuffTracker:SetMovable(true)
		debuffTracker:SetClampedToScreen(true)
		debuffTracker:SetScript("OnMouseDown", function(self) self:StartMoving(); self:SetBackdropColor(0, 0, 0, .6); end)
		debuffTracker:SetScript("OnMouseUp", function(self)
			self:StopMovingOrSizing()
			if (Elementarist.locked) then
				self:SetBackdropColor(0, 0, 0, 0)
			else
				self:SetBackdropColor(0, 0, 0, .3)
			end
			local _,_,rp,x,y = self:GetPoint()
			ElementaristDB.debuffx = x
			ElementaristDB.debuffy = y
			ElementaristDB.debuffrelativePoint = rp
		end)
		debuffTracker:SetScript("OnDragStop", function(self)
			self:StopMovingOrSizing();
			if (Elementarist.locked) then
				self:SetBackdropColor(0, 0, 0, 0)
			else
				self:SetBackdropColor(0, 0, 0, .3)
			end
			local _,_,rp,x,y = self:GetPoint()
			ElementaristDB.debuffx = x
			ElementaristDB.debuffy = y
			ElementaristDB.debuffrelativePoint = rp
		end)
	end
	debuffTracker:SetPoint(ElementaristDB.debuffrelativePoint,ElementaristDB.debuffx,ElementaristDB.debuffy)

	t = displayFrame_dps:CreateFontString("$parentDebuffText","OVERLAY","GameFontNormalLarge");
	t:SetAllPoints(debuffTracker)
	t:SetAlpha(1)
	t:SetText("")
	Elementarist.textList["debuff"] = t

	local debuffTracker_cd = CreateFrame("Frame","$parent_debuff", ElementaristDebuffTrackerFrame)
	debuffTracker_cd:SetWidth(50)
	debuffTracker_cd:SetHeight(50)
	debuffTracker_cd:SetPoint("CENTER", 0, 0)
	t = debuffTracker_cd:CreateTexture(nil,"BACKGROUND")
	t:SetTexture(nil)
	t:SetAllPoints(debuffTracker_cd)
	t:SetAlpha(1)
	debuffTracker_cd.texture = t
	Elementarist.textureList["debuff"] = t

	local debuffCooldownFrame = CreateFrame("Cooldown","$parent_debuffcooldown", ElementaristDebuffTrackerFrame, "CooldownFrameTemplate")
	debuffCooldownFrame:SetHeight(50)
	debuffCooldownFrame:SetWidth(50)
	debuffCooldownFrame:ClearAllPoints()
	debuffCooldownFrame:SetPoint("CENTER", debuffTracker, "CENTER", 0, 0)

	debuffTracker:SetAlpha(ElementaristDB.debuffalpha)

	Elementarist.debuffTracker = debuffTracker
	Elementarist.debuffCooldownFrame = debuffCooldownFrame

	DEFAULT_CHAT_FRAME:AddMessage("Elementarist " .. Elementarist.versionNumber .. " loaded")
end

function Elementarist:GetSpellCooldownRemaining(spell)
	local s, d, _ = GetSpellCooldown(spell)
	if (d) and (d>0) then
		d = s - GetTime() + d
	end

	return d
end

function Elementarist:hasDeBuff(unit, spellName, casterUnit)
	local i = 1;
	while true do
		local name, rank, icon, count, debuffType, duration, expirationTime, unitCaster, isStealable = UnitDebuff(unit, i);
		if not name then
			break;
		end
		if string.match(name, spellName) and ((unitCaster == casterUnit) or (casterUnit == nil)) then
	   		return name, rank, icon, count, debuffType, duration, expirationTime, unitCaster, isStealable;
		end
		i = i + 1;
	end
end

function Elementarist:hasBuff(unit, spellName, stealableOnly, getByID)
	local i = 1;
	while true do
		local name, rank, icon, count, buffType, duration, expirationTime, source, isStealable, shouldConsolidate, spellId = UnitBuff(unit, i);
		if not name then
			break;
		end
		if (not getByID) then
			if string.match(name, spellName) then
				if (not stealableOnly) or (isStealable) then
					return name, rank, icon, count, buffType, duration, expirationTime, unitCaster, isStealable;
				end
			end
		else
			if (getByID == spellId) then
				return name, rank, icon, count, buffType, duration, expirationTime, unitCaster, isStealable;
			end
		end
		i = i + 1;
	end
end

function Elementarist:hasTotem(unit, spellName)
	local i = 1;
	while true do
		local name, rank, icon, count, buffType, duration, expirationTime, source, isStealable = UnitBuff(unit, i);
		if not name then
			break;
		end
		if (string.match(name, spellName) or (string.match(icon, spellName))) and (expirationTime==0) then
	   		return name, rank, icon, count, buffType, duration, expirationTime, unitCaster, isStealable;
		end
		i = i + 1;
	end
end

function Elementarist:SpellAvailable(spell)
	if (not spell) then
		return false
	end
	local u,m = IsUsableSpell(spell)
	if (u) then
		return true
	else
		return false
	end
end

function Elementarist:NextSpell(timeshift,exspell1,exspell2)
	local guid = UnitGUID("target")
	local currentTime = GetTime()
	local s,d,e
	local name, rank, icon, count, debuffType, duration, fsExpiration, unitCaster
	local lastSpell
	
	if (exspell1) then
		if (exspell2) then
			lastSpell = exspell2
		else
			lastSpell = exspell1
		end
	else
		lastSpell = Elementarist.lastSpell
	end
	
	Elementarist.lastBaseGCD = 1.5 - (1.5 * Elementarist.spellHaste * .01)
	
	local flameshockavail = false
	local LvBct = 2 - (2 * Elementarist.spellHaste * .01)

	-- if target is dead, return ""
	if (UnitHealth("target")<=0) then
		return ""
	end
	
	if (not timeshift) then
		timeshift = 0
	end
	
	-- check current spell
	local spellInCast,_,_,_,sICstartTime,sICendTime = UnitCastingInfo("player")
	if (sepllInCast) then
		if ( (sICendTime - sICstartTime) / 1000 ) < Elementarist.lastBaseGCD then
			sICendTime = sICstartTime + (Elementarist.lastBaseGCD * 1000)
		end
		timeshift = timeshift + (sICendTime / 1000) - GetTime()
	else
		-- no spell in cast, check global cd via Flametongue Weapon
		if (Elementarist.SpellList["Flametongue Weapon"]) then
			local ftcd = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Flametongue Weapon"])
			if (ftcd) then
				timeshift = timeshift + Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Flametongue Weapon"])
			else
				timeshift = timeshift + Elementarist.lastBaseGCD
			end
		else
			timeshift = timeshift + Elementarist.lastBaseGCD
		end
	end

	-- check if Flame shock applied on target first
	if (exspell1 ~= Elementarist.SpellList["Flame Shock"]) and (exspell2 ~= Elementarist.SpellList["Flame Shock"]) then
		if IsSpellInRange(Elementarist.SpellList["Flame Shock"], "target") == 1 then
			name, rank, icon, count, debuffType, duration, fsExpiration, unitCaster = Elementarist:hasDeBuff("target", Elementarist.SpellList["Flame Shock"], "player");
			if (not fsExpiration) then
				fsExpiration = 0
			end
			d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Flame Shock"])
			if ((d - timeshift) <= 0) then
				flameshockavail = true
				local doFS = false
				if (unitCaster ~= "player") then	-- fs debuff is not casted by the player
					name = false
					fsExpiration = 0
				end
				if (not name) then 	-- no fs debuff on target
					fsExpiration = 0
				end
				if (ElementaristDB.Behaviour == Elementarist.Behaviours["1"]) then
					doFS = true
				else	-- if fs before lvb, check lvb cd
					d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Lava Burst"])
					if (d <= (timeshift + Elementarist.lastBaseGCD)) then
						doFS = true
					end
				end
				if (doFS) and ((fsExpiration - GetTime() - timeshift) < 0) then
					return Elementarist.SpellList["Flame Shock"]
				end
			end
		end
	end
	
	if (not fsExpiration) then
		fsExpiration = 0
	end
	if (exspell1 ~= Elementarist.SpellList["Lava Burst"]) and (exspell2 ~= Elementarist.SpellList["Lava Burst"]) then
		if (IsSpellInRange(Elementarist.SpellList["Lava Burst"], "target") == 1) and
		(
			((fsExpiration~=0) and ((fsExpiration-GetTime()-timeshift) > LvBct)) or 
			(exspell1 == Elementarist.SpellList["Flame Shock"]) or 
			(exspell2 == Elementarist.SpellList["Flame Shock"])
		) then
			if ((Elementarist.SpellList["Lava Burst"]) ~= spellInCast) then
				d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Lava Burst"])
				if ((d-timeshift) <= 0) then
					return Elementarist.SpellList["Lava Burst"]
				end
			end
		end
	end
	
	-- if >=3 foes are available, and Fire Nova enabled in settings, and not in cd (and has Fire totem???)
	if (exspell1 ~= Elementarist.SpellList["Fire Nova"]) and (exspell2 ~= Elementarist.SpellList["Fire Nova"]) then
		if (ElementaristDB.FireNova) and (Elementarist.person["foeCount"]>=3) then
			local haveFireTotem,fireTotemName,_,_ = GetTotemInfo(1)
			if (haveFireTotem) and (fireTotemName ~= "") then
				d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Fire Nova"])
				if ((d-timeshift) <= 0) then
					return Elementarist.SpellList["Fire Nova"]
				end
			end
		end
	end

	-- CL if spellpower is below 1920, and not on CD
	-- or there are multiple targets, (count in a dirty way from combat log, not to accurate!!!)
	if (exspell1 ~= Elementarist.SpellList["Chain Lightning"]) and (exspell2 ~= Elementarist.SpellList["Chain Lightning"]) then
		if (
			(ElementaristDB.CLBehaviour == Elementarist.CLBehaviours["2"]) or
			((ElementaristDB.CLBehaviour == Elementarist.CLBehaviours["1"]) and (lastSpell == Elementarist.SpellList["Lava Burst"]))
		) or (Elementarist.person["foeCount"]>1) then
			if IsSpellInRange(Elementarist.SpellList["Chain Lightning"], "target") == 1 then
				if ((Elementarist.SpellList["Chain Lightning"]) ~= spellInCast) then
					d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Chain Lightning"])
					if ((d-timeshift) <= 0) then
						return Elementarist.SpellList["Chain Lightning"]
					end
				end
			end
		end
	end

	-- otherwise lightning bolt
	if IsSpellInRange(Elementarist.SpellList["Lightning Bolt"], "target") == 1 then
		return Elementarist.SpellList["Lightning Bolt"]
	end
	
	-- if nothing works, try flameshock again
	if flameshockavail then
		return Elementarist.SpellList["Flame Shock"]
	end
	
	return ""
end

function Elementarist:MiscSpell()
	-- Miscelaneous spell order:
	-- Flametongoue weapon
	-- Water Shield
	-- Totem of Wrath
	-- Wrath of Air totem
	-- Elemental Mastery
	-- Berserking troll racial (if available)

	local d, e
	local name, rank, icon, count, debuffType, duration, expirationTime, unitCaster
	
	-- Alchemy flask of the north
	if GetItemCount(Elementarist.CustomIDs["Flask of the North Item"]) ~= 0 then
		name, rank, icon, count, debuffType, duration, expirationTime, unitCaster = Elementarist:hasBuff("player", Elementarist.SpellList["Flask of the Frost Wyrm"]);
		if (name == nil) then
			name, rank, icon, count, debuffType, duration, expirationTime, unitCaster = Elementarist:hasBuff("player", Elementarist.SpellList["Flask of the North"], false, Elementarist.CustomIDs["Flask of the North Spell"]);
			if (name == nil) or (expirationTime < 2) then
				if (icon == nil) then
					icon = GetItemIcon(Elementarist.CustomIDs["Flask of the North Item"]) 
				end
				return nil,icon
			end
		end
	end

	-- Flametongue weapon
	if Elementarist:SpellAvailable(Elementarist.SpellList["Flametongue Weapon"]) then
		local hasMainHandEnchant, mainHandExpiration, _, _, _, _ = GetWeaponEnchantInfo()
		if (hasMainHandEnchant == nil) or ((mainHandExpiration / 60000) < 1) then
			return Elementarist.SpellList["Flametongue Weapon"]
		end
	end

	-- Water Shield
	if Elementarist:SpellAvailable(Elementarist.SpellList["Water Shield"]) then
		name, rank, icon, count, debuffType, duration, expirationTime, unitCaster = Elementarist:hasBuff("player", Elementarist.SpellList["Water Shield"]);
		if (name == nil) then
			return Elementarist.SpellList["Water Shield"]
		end
	end

	-- Totem of Wrath
	if Elementarist:SpellAvailable(Elementarist.SpellList["Totem of Wrath Totem"]) then
		name = Elementarist:hasBuff("player",Elementarist.SpellList["Demonic Pact"])
		if (name == nil) then
			name, rank, icon, count, debuffType, duration, expirationTime, unitCaster = Elementarist:hasTotem("player", Elementarist.SpellList["Totem of Wrath Totem"]);
			if (name == nil) or (duration > 0) then
				return Elementarist.SpellList["Totem of Wrath Totem"]
			end
		end
		name = Elementarist:hasDeBuff("target",Elementarist.SpellList["Heart of the Crusader"],nil)
		if (name == nil) then
			name, _, icon, _, _, d, e = Elementarist:hasDeBuff("target",Elementarist.SpellList["Totem of Wrath Debuff"],"player")
			if (not name) then
				return Elementarist.SpellList["Totem of Wrath Totem"]
			end
		end
	end

	-- Wrath of Air totem
	if Elementarist:SpellAvailable(Elementarist.SpellList["Wrath of Air Totem"]) then
		name, rank, icon, count, debuffType, duration, expirationTime, unitCaster = Elementarist:hasTotem("player", Elementarist.SpellList["Wrath of Air Totem"]);
		if (name == nil) then
			return Elementarist.SpellList["Wrath of Air Totem"]
		end
	end

	-- Elemental Mastery
	if Elementarist:SpellAvailable(Elementarist.SpellList["Elemental Mastery"]) then
		d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Elemental Mastery"])
		if d <= 0.5 then
			return Elementarist.SpellList["Elemental Mastery"]
		end
	end

	-- Berserking
	if Elementarist:SpellAvailable(Elementarist.SpellList["Berserking"]) then
		d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Berserking"])
		if d <= 0.5 then
			return Elementarist.SpellList["Berserking"]
		end
	end

	-- Blood Fury
	if Elementarist:SpellAvailable(Elementarist.SpellList["Blood Fury"]) then
		d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Blood Fury"])
		if d <= 0.5 then
			return Elementarist.SpellList["Blood Fury"]
		end
	end
	
	return ""

end

function Elementarist:IntSpell()
	-- interruptions, mana recharge wia thunderstorm, and purge on target
	if Elementarist:SpellAvailable(Elementarist.SpellList["Wind Shear"]) then
		if IsSpellInRange(Elementarist.SpellList["Wind Shear"], "target") == 1 then
			local name = UnitCastingInfo("target")
			local channel = UnitChannelInfo("target")
			local _, status, threatpct, _, _ = UnitDetailedThreatSituation("player", "target")
			if ((name) or (channel)) or ((status) and (threatpct>80) and (Elementarist.person["friendCount"]>1) and (Elementarist.inParty>0)) then
				local d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Wind Shear"])
				if (d) and (d<0.5) and (IsSpellInRange(Elementarist.SpellList["Wind Shear"], "target") ) then
					return Elementarist.SpellList["Wind Shear"]
				end
			end
		
			local name, subText, text, texture, startTime, endTime, isTradeSkill, castID, interruptable = UnitChannelInfo("target")
			if (name) then
				local d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Wind Shear"])
				if (d<0.5) and (interruptable) and (IsSpellInRange(Elementarist.SpellList["Wind Shear"], "target") ) then
					return Elementarist.SpellList["Wind Shear"]
				end
			end
		end
	end
	
	-- check if mana is below 70% and Thunderstorm available
	if Elementarist:SpellAvailable(Elementarist.SpellList["Thunderstorm"]) then
		local d = GetSpellCooldown(Elementarist.SpellList["Thunderstorm"])
		if (d) and ((UnitPower("player",0) / UnitPowerMax("player",0)) < 0.7) and (d < 0.5) then
			return Elementarist.SpellList["Thunderstorm"]
		end
	end
	
	-- check if purgeable buff is on target (not sure if this is ok)
	if Elementarist:SpellAvailable(Elementarist.SpellList["Purge"]) then
		if IsSpellInRange(Elementarist.SpellList["Purge"], "target") == 1 then
			local name, rank, icon, count, debuffType, duration, expirationTime, unitCaster = Elementarist:hasBuff("target", ".", 1);
			if (name) then
				local d = Elementarist:GetSpellCooldownRemaining(Elementarist.SpellList["Purge"])
				if (d<0.5) then
					return Elementarist.SpellList["Purge"]
				end
			end
		end
	end

	return ""
end

function Elementarist:DecideSpells()

	Elementarist.timeSinceLastUpdate = 0;
	local currentTime = GetTime()

	local guid = UnitGUID("target")
	local guid = UnitGUID("target")
	if  UnitName("target") == nil or UnitIsFriend("player","target") ~= nil or UnitHealth("target") == 0 then
		guid = nil
	end

	if UnitInVehicle("player") then
		-- player is in a "vehicle" don't suggest spell
		Elementarist.textureList["next"]:SetTexture(nil)
		Elementarist.textureList["next1"]:SetTexture(nil)
		Elementarist.textureList["next2"]:SetTexture(nil)
		Elementarist.textureList["Misc"]:SetTexture(nil)
		Elementarist.textureList["int"]:SetTexture(nil)
	
		return
	end

	if guid == nil then
		Elementarist.textureList["next"]:SetTexture(nil)
		Elementarist.textureList["next1"]:SetTexture(nil)
		Elementarist.textureList["next2"]:SetTexture(nil)
		Elementarist.textureList["Misc"]:SetTexture(nil)
		Elementarist.textureList["int"]:SetTexture(nil)
	
		return
	end 
  	if (UnitHealth("target") == 0) then
		Elementarist.textureList["next"]:SetTexture(nil)
		Elementarist.textureList["next1"]:SetTexture(nil)
		Elementarist.textureList["next2"]:SetTexture(nil)

		return
  	end

	local spell = ""
	spell = Elementarist:NextSpell()
	local d = Elementarist:GetSpellCooldownRemaining(spell)
	if (d) and (d>0) then
		local cdStart = currentTime - Elementarist.lastBaseGCD + d  -- should be less then the base gcd if we are suggesting it
		if (cdStart) and (Elementarist.lastBaseGCD) then
			Elementarist.cooldownFrame:SetCooldown(cdStart, Elementarist.lastBaseGCD)
		end
	end
	Elementarist.textureList["next"]:SetTexture(GetSpellTexture(spell))
	
	local _,_,_,_,_,_,ct1=GetSpellInfo(spell)
	if (not ct1) then
		ct1 = 0
	else
		ct1 = (ct1 / 1000)
	end
	local spell1 = Elementarist:NextSpell(ct1,spell)
	Elementarist.textureList["next1"]:SetTexture(GetSpellTexture(spell1))

	local _,_,_,_,_,_,ct2=GetSpellInfo(spell1)
	if (not ct2) then
		ct2 = 0
	else
		ct2 = (ct2 / 1000)
	end
	if (not ct2) or (ct2 < Elementarist.lastBaseGCD) then
		ct2 = Elementarist.lastBaseGCD
	end
	local spell2 = Elementarist:NextSpell(ct1+ct2,spell,spell1)
	Elementarist.textureList["next2"]:SetTexture(GetSpellTexture(spell2))
	
	local icon

	spell,icon = Elementarist:MiscSpell()

	if (icon) then
		Elementarist.textureList["Misc"]:SetTexture(icon)
	else
		if (spell) then
			Elementarist.textureList["Misc"]:SetTexture(GetSpellTexture(spell))
		end
	end

	spell = Elementarist:IntSpell()
	Elementarist.textureList["int"]:SetTexture(GetSpellTexture(spell))


end

function Elementarist:OnUpdate(elapsed)
	if (Elementarist:isEnabled()) then
		Elementarist.timeSinceLastUpdate = Elementarist.timeSinceLastUpdate + elapsed 
		
		if (Elementarist.timeSinceLastUpdate > (1.5 - (1.5 * Elementarist.spellHaste * .01)) * 0.3) then
			Elementarist:DecideSpells()
		end
		if (not ElementaristDB.debuffdisabled) then
			Elementarist.debuffTrackerUpdate = Elementarist.debuffTrackerUpdate + elapsed 
			if (
				((Elementarist.OmniCC) and (Elementarist.debuffTrackerUpdate >= 1)) or 
				((not Elementarist.OmniCC) and (Elementarist.debuffTrackerUpdate >= 1))
			) then
				Elementarist:UpdateDebuffTracker()
			end
		end
	end
end

function Elementarist:CreateCheckButton(name, parent, table, field, radio)
	local button
	if radio then
		button = CreateFrame('CheckButton', parent:GetName() .. name, parent, 'SendMailRadioButtonTemplate')
	else
		button = CreateFrame('CheckButton', parent:GetName() .. name, parent, 'OptionsCheckButtonTemplate')
	end
	local frame = _G[button:GetName() .. 'Text']
	frame:SetText(name)
	frame:SetTextColor(1, 1, 1, 1)
	frame:SetFontObject(GameFontNormal)
	button:SetScript("OnShow", 
		function (self) 
			self:SetChecked(table[field]) 
			self.origValue = table[field] or self.origValue
		end 
	)
	if radio then
		button:SetScript("OnClick", 
			function (self, button, down)
				this:SetChecked(1)
				table[field] = not table[field]
			end 
		)
	else
		button:SetScript("OnClick", 
			function (self, button, down) 
				table[field] = not table[field]
			end
		)
	end

	function button:Restore() 
		table[field] = self.origValue 
	end 
	return button 
end

function Elementarist:CreateSlider(text, parent, low, high, step)
	local name = parent:GetName() .. text
	local slider = CreateFrame('Slider', name, parent, 'OptionsSliderTemplate')
	slider:SetScript('OnMouseWheel', Slider_OnMouseWheel)
	slider:SetMinMaxValues(low, high)
	slider:SetValueStep(step)
	slider:EnableMouseWheel(true)
	_G[name .. 'Text']:SetText(text)
	_G[name .. 'Low']:SetText('')
	_G[name .. 'High']:SetText('')
	local text = slider:CreateFontString(nil, 'BACKGROUND')
	text:SetFontObject('GameFontHighlightSmall')
	text:SetPoint('LEFT', slider, 'RIGHT', 7, 0)
	slider.valText = text
	return slider
end

function Elementarist:CreateButton(text, parent)
	local name = parent:GetName() .. text
	local button = CreateFrame('Button', name, parent, 'UIPanelButtonTemplate')
	_G[name .. 'Text']:SetText(text)
	local text = button:CreateFontString(nil, 'BACKGROUND')
	text:SetFontObject('GameFontHighlightSmall')
	text:SetPoint('LEFT', button, 'RIGHT', 7, 0)
	button.valText = text
	return button
end

function Elementarist:CreateDropDownMenu(text, parent, dbTree, varName, itemList, width)
	local name = parent:GetName() .. text
    local menu = CreateFrame("Frame", name, parent, "UIDropDownMenuTemplate");
    menu.displayMode = "MENU"

	local frame = _G[menu:GetName() .. 'Text']
	frame:SetText(text)
	frame:SetTextColor(1, 1, 1, 1)
	frame:SetFontObject(GameFontNormal)

    menu:EnableMouse(true);
    if(width) then
        _G.UIDropDownMenu_SetWidth(menu, width);
    end
    menu.itemList = itemList or {};
    menu.init = function()
            for i=1, #menu.itemList do
                if(not menu.itemList[i].hooked) then
                    local func = menu.itemList[i].func or function(self) end;
                    menu.itemList[i].func = function(self, arg1, arg2)
                        self = self or _G.this; -- wotlk/tbc hack
                        dbTree[varName] = self.value;
                        _G.UIDropDownMenu_SetSelectedValue(menu, self.value);
                        func(self, arg1, arg2);
                    end
                    menu.itemList[i].hooked = true;
                end
                local info = _G.UIDropDownMenu_CreateInfo();
                for k,v in pairs(menu.itemList[i]) do
                    info[k] = v;
                end
                _G.UIDropDownMenu_AddButton(info, _G.UIDROPDOWNMENU_MENU_LEVEL);
            end
        end
    menu:SetScript("OnShow", function(self)
            _G.UIDropDownMenu_Initialize(self, self.init);
            _G.UIDropDownMenu_SetSelectedValue(self, dbTree[varName]);
        end);
    menu.SetValue = function(self, value)
            _G.UIDropDownMenu_SetSelectedValue(self, value);
        end;
    menu:Hide(); menu:Show();
    return menu;
end

function Elementarist:ApplySettings()
	Elementarist:InitSettings()
	if (not ElementaristDB.locked) then
		Elementarist.displayFrame:EnableMouse(true)
		Elementarist.displayFrame:SetMovable(true)
		Elementarist.displayFrame:SetClampedToScreen(true)
		Elementarist.displayFrame:SetScript("OnMouseDown", function(self) self:StartMoving(); self:SetBackdropColor(0, 0, 0, .4); end)
		Elementarist.displayFrame:SetScript("OnMouseUp", function(self)
			self:StopMovingOrSizing()
			if (Elementarist.locked) then
				self:SetBackdropColor(0, 0, 0, 0)
			else
				self:SetBackdropColor(0, 0, 0, .3)
			end
			local _,_,rp,x,y = self:GetPoint()
			ElementaristDB.x = x
			ElementaristDB.y = y
			ElementaristDB.relativePoint = rp
		end)
		Elementarist.displayFrame:SetScript("OnDragStop", function(self)
			self:StopMovingOrSizing();
			if (Elementarist.locked) then
				self:SetBackdropColor(0, 0, 0, 0)
			else
				self:SetBackdropColor(0, 0, 0, .3)
			end
			local _,_,rp,x,y = self:GetPoint()
			ElementaristDB.x = x
			ElementaristDB.y = y
			ElementaristDB.relativePoint = rp
		end)
		Elementarist.displayFrame:SetBackdropColor(0, 0, 0, .3)
		Elementarist.debuffTracker:EnableMouse(true)
		Elementarist.debuffTracker:SetMovable(true)
		Elementarist.debuffTracker:SetClampedToScreen(true)
		Elementarist.debuffTracker:SetScript("OnMouseDown", function(self) self:StartMoving(); self:SetBackdropColor(0, 0, 0, .4); end)
		Elementarist.debuffTracker:SetScript("OnMouseUp", function(self)
			self:StopMovingOrSizing()
			if (Elementarist.locked) then
				self:SetBackdropColor(0, 0, 0, 0)
			else
				self:SetBackdropColor(0, 0, 0, .3)
			end
			local _,_,rp,x,y = self:GetPoint()
			ElementaristDB.debuffx = x
			ElementaristDB.debuffy = y
			ElementaristDB.debuffrelativePoint = rp
		end)
		Elementarist.debuffTracker:SetScript("OnDragStop", function(self)
			self:StopMovingOrSizing();
			if (Elementarist.locked) then
				self:SetBackdropColor(0, 0, 0, 0)
			else
				self:SetBackdropColor(0, 0, 0, .3)
			end
			local _,_,rp,x,y = self:GetPoint()
			ElementaristDB.debuffx = x
			ElementaristDB.debuffy = y
			ElementaristDB.debuffrelativePoint = rp
		end)
		Elementarist.debuffTracker:SetBackdropColor(0, 0, 0, .3)
	else
		if (Elementarist.displayFrame) then
			Elementarist.displayFrame:EnableMouse(false)
			Elementarist.displayFrame:SetMovable(false)
			Elementarist.displayFrame:SetBackdropColor(0, 0, 0, .0)
			Elementarist.debuffTracker:EnableMouse(false)
			Elementarist.debuffTracker:SetMovable(false)
			Elementarist.debuffTracker:SetBackdropColor(0, 0, 0, .0)
		end
	end
	if (not Elementarist:isEnabled()) then
		if (Elementarist.displayFrame) then
			Elementarist.displayFrame:Hide()
			Elementarist.debuffTracker:Hide()
		end
	else
		if (Elementarist.displayFrame) then
			Elementarist.displayFrame:Show()
			if (ElementaristDB.debuffdisabled) then
				Elementarist.debuffTracker:Hide()
			else
				Elementarist.debuffTracker:Show()
			end
		end
	end
	if (Elementarist.displayFrame) then
		Elementarist.displayFrame:SetAlpha(ElementaristDB.alpha)
		Elementarist.displayFrame:SetScale(ElementaristDB.scale)
		Elementarist.debuffTracker:SetAlpha(ElementaristDB.debuffalpha)
		Elementarist.debuffTracker:SetScale(ElementaristDB.debuffscale)
	end
end

function Elementarist:StoreUIValues()
	Elementarist.prevDB["scale"] = ElementaristDB.scale
	Elementarist.prevDB["debuffscale"] = ElementaristDB.debuffscale
	Elementarist.prevDB["alpha"] = ElementaristDB.alpha
	Elementarist.prevDB["debuffalpha"] = ElementaristDB.debuffalpha
	Elementarist.prevDB["FireNova"] = ElementaristDB.FireNova
	Elementarist.prevDB["Behaviour"] = ElementaristDB.Behaviour
	Elementarist.prevDB["CLBehaviour"] = ElementaristDB.CLBehaviour
	Elementarist.prevDB["locked"] = ElementaristDB.locked
	Elementarist.prevDB["ThreatWarning"] = ElementaristDB.ThreatWarning
	Elementarist.prevDB["enabled"] = ElementaristDB.enabled
	Elementarist.prevDB["disableIfNotEle"] = ElementaristDB.disableIfNotEle
	Elementarist.prevDB["debuffdisabled"] = ElementaristDB.debuffdisabled
	Elementarist.prevDB["x"] = ElementaristDB.x
	Elementarist.prevDB["y"] = ElementaristDB.y
	Elementarist.prevDB["debuffx"] = ElementaristDB.debuffx
	Elementarist.prevDB["debuffy"] = ElementaristDB.debuffy
	Elementarist.prevDB["relativePoint"] = ElementaristDB.relativePoint
	Elementarist.prevDB["debuffrelativePoint"] = ElementaristDB.debuffrelativePoint
end

function Elementarist:ReStoreUIValues()
	ElementaristDB.scale = Elementarist.prevDB["scale"]
	ElementaristDB.alpha = Elementarist.prevDB["alpha"]
	ElementaristDB.debuffscale = Elementarist.prevDB["debuffscale"]
	ElementaristDB.debuffalpha = Elementarist.prevDB["debuffalpha"]
	ElementaristDB.FireNova = Elementarist.prevDB["FireNova"]
	ElementaristDB.Behaviour = Elementarist.prevDB["Behaviour"]
	ElementaristDB.CLBehaviour = Elementarist.prevDB["CLBehaviour"]
	ElementaristDB.ThreatWarning = Elementarist.prevDB["ThreatWarning"]
	ElementaristDB.locked = Elementarist.prevDB["locked"]
	ElementaristDB.enabled = Elementarist.prevDB["enabled"]
	ElementaristDB.disableIfNotEle = Elementarist.prevDB["disableIfNotEle"]
	ElementaristDB.debuffdisabled = Elementarist.prevDB["debuffdisabled"]
	ElementaristDB.x = Elementarist.prevDB["x"] 
	ElementaristDB.y = Elementarist.prevDB["y"]
	ElementaristDB.debuffx = Elementarist.prevDB["debuffx"] 
	ElementaristDB.debuffy = Elementarist.prevDB["debuffy"]
	ElementaristDB.relativePoint = Elementarist.prevDB["relativePoint"]
	ElementaristDB.debuffrelativePoint = Elementarist.prevDB["debuffrelativePoint"]
end

function Elementarist:CreateConfig()
	Elementarist.configPanel = CreateFrame( "Frame", "ElementaristConfigPanel", UIParent );
	-- Register in the Interface Addon Options GUI
	-- Set the name for the Category for the Options Panel
	Elementarist.configPanel.name = "Elementarist";
	
	local EnableBtn = Elementarist:CreateCheckButton(L.CONFIG_ENABLED, Elementarist.configPanel, ElementaristDB, "enabled", false)
	EnableBtn:SetPoint('TOPLEFT', 10, -8) 

	local EnableBtn = Elementarist:CreateCheckButton(L.CONFIG_ONLY_ON_ELE, Elementarist.configPanel, ElementaristDB, "disableIfNotEle", false)
	EnableBtn:SetPoint('TOPLEFT', 40, -28) 

	local LockBtn = Elementarist:CreateCheckButton(L.CONFIG_LOCK_FRAMES, Elementarist.configPanel, ElementaristDB, "locked", false)
	LockBtn:SetPoint('TOPLEFT', 10, -58) 

	local Scale = Elementarist:CreateSlider(L.CONFIG_SPELL_ADV_SCALE, Elementarist.configPanel, .25, 3, .1)
	Scale:SetScript('OnShow', function(self)
		self.onShow = true
		Elementarist:StoreUIValues()
		self:SetValue(ElementaristDB.scale)
		self.onShow = nil
	end)
	Scale:SetScript('OnValueChanged', function(self, value)
		self.valText:SetText(format('%.1f', value))
		if not self.onShow then
			ElementaristDB.scale=value
			Elementarist.displayFrame:SetScale(value)
			
		end
	end)
	Scale:SetPoint("TOPLEFT",10,-98)
	Scale:Show()

	local Alpha = Elementarist:CreateSlider(L.CONFIG_SPELL_ADV_ALPHA, Elementarist.configPanel, .0, 1, .1)
	Alpha:SetScript('OnShow', function(self)
		self.onShow = true
		self:SetValue(ElementaristDB.alpha)
		self.onShow = nil
	end)
	Alpha:SetScript('OnValueChanged', function(self, value)
		self.valText:SetText(format('%.1f', value))
		if not self.onShow then
			ElementaristDB.alpha=value
			Elementarist.displayFrame:SetAlpha(value)
		end
	end)
	Alpha:SetPoint("TOPLEFT",10,-138)
	Alpha:Show()

	local DebuffDisableBtn = Elementarist:CreateCheckButton(L.CONFIG_DISABLE_DEBUFF_TRACKER, Elementarist.configPanel, ElementaristDB, "debuffdisabled", false)
	DebuffDisableBtn:SetPoint('TOPLEFT', 10, -158) 

	local DebuffScale = Elementarist:CreateSlider(L.CONFIG_DEBUFF_TRACKER_SCALE, Elementarist.configPanel, .25, 3, .1)
	DebuffScale:SetScript('OnShow', function(self)
		self.onShow = true
		Elementarist:StoreUIValues()
		self:SetValue(ElementaristDB.debuffscale)
		self.onShow = nil
	end)
	DebuffScale:SetScript('OnValueChanged', function(self, value)
		self.valText:SetText(format('%.1f', value))
		if not self.onShow then
			ElementaristDB.debuffscale=value
			Elementarist.debuffTracker:SetScale(value)
			
		end
	end)
	DebuffScale:SetPoint("TOPLEFT",10,-198)
	DebuffScale:Show()

	local DebuffAlpha = Elementarist:CreateSlider(L.CONFIG_DEBUFF_TRACKER_ALPHA, Elementarist.configPanel, .0, 1, .1)
	DebuffAlpha:SetScript('OnShow', function(self)
		self.onShow = true
		self:SetValue(ElementaristDB.debuffalpha)
		self.onShow = nil
	end)
	DebuffAlpha:SetScript('OnValueChanged', function(self, value)
		self.valText:SetText(format('%.1f', value))
		if not self.onShow then
			ElementaristDB.debuffalpha=value
			Elementarist.debuffTracker:SetAlpha(value)
		end
	end)
	DebuffAlpha:SetPoint("TOPLEFT",10,-238)
	DebuffAlpha:Show()

	local FireNovaBtn = Elementarist:CreateCheckButton(L.CONFIG_ENABLE_NOVA, Elementarist.configPanel, ElementaristDB, "FireNova", false)
	FireNovaBtn:SetPoint('TOPLEFT', 10, -258) 

	local FSBehaviourMenu = Elementarist:CreateDropDownMenu(L.CONFIG_BEHAVIOUR, Elementarist.configPanel, ElementaristDB, "Behaviour", {{text = Elementarist.Behaviours["1"]}, {text = Elementarist.Behaviours["2"]}},200)
	FSBehaviourMenu:SetPoint('TOPLEFT', 10, -288) 

	local CLBehaviourMenu = Elementarist:CreateDropDownMenu(L.CONFIG_CLSTBEHAVIOUR, Elementarist.configPanel, ElementaristDB, "CLBehaviour", {{text = Elementarist.CLBehaviours["0"]}, {text = Elementarist.CLBehaviours["1"]}, {text = Elementarist.CLBehaviours["2"]}},200)
	CLBehaviourMenu:SetPoint('TOPLEFT', 10, -318) 

	local ThreatWarnBtn = Elementarist:CreateCheckButton(L.CONFIG_THREAT_WARNING, Elementarist.configPanel, ElementaristDB, "ThreatWarning", false)
	ThreatWarnBtn:SetPoint('TOPLEFT', 10, -348) 

	local ResetBtn = Elementarist:CreateButton(L.CONFIG_RESET_POSITIONS, Elementarist.configPanel)
	ResetBtn:SetWidth(160)
	ResetBtn:SetHeight(22)
	ResetBtn:SetScript('OnClick', function()
		Elementarist:ResetPosition()
	end)
	ResetBtn:SetPoint("TOPLEFT",10,-378)
	ResetBtn:Show()
	
	Elementarist.configPanel.okay = function()
		Elementarist:ApplySettings()
	end
	Elementarist.configPanel.cancel = function()
		-- cancel button pressed, revert changes
		Elementarist:ReStoreUIValues()
		Elementarist:ApplySettings()
	end
	Elementarist.configPanel.default = function()
		-- default button pressed, reset setting
		ElementaristDB.scale = 1
		ElementaristDB.debuffscale = 1
		ElementaristDB.locked = false
		ElementaristDB.enabled = true
		ElementaristDB.disableIfNotEle = true
		ElementaristDB.debuffdisabled = false
		ElementaristDB.alpha = 0.8
		ElementaristDB.debuffalpha = 1
		ElementaristDB.FireNova = true
		ElementaristDB.Behaviour = Elementarist.Behaviours["1"]
		ElementaristDB.CLBehaviour = Elementarist.CLBehaviours["0"]
		ElementaristDB.ThreatWarning = true
		ElementaristDB.x = -200
		ElementaristDB.y = -200
		ElementaristDB.relativePoint = "CENTER"
		ElementaristDB.x = -200
		ElementaristDB.y = -100
		ElementaristDB.relativePoint = "CENTER"
	end

	-- always show frame if config panel is open
	Elementarist.configPanel:SetScript('OnShow', function(self)
		self.onShow = true
		Elementarist:DecideSpells()
		self.onShow = nil
	end)
	Elementarist.configPanel:SetScript('OnHide', function(self)
		self.onHide = true
		Elementarist:DecideSpells()
		self.onHide = nil
	end)
	-- Add the panel to the Interface Options
	InterfaceOptions_AddCategory(Elementarist.configPanel);
end

function Elementarist.Options()
  InterfaceOptionsFrame_OpenToCategory(getglobal("ElementaristConfigPanel"))
end