local GetTime = GetTime;
local FormatTime = AzCastBar.FormatTime;

-- Extra Options
local extraOptions = {
	{
		[0] = "Additional",
		{ type = "Color", var = "colNormal", default = { 0.4, 0.6, 0.8 }, label = "Cooldown Bar Color", y = 12 },
		{ type = "Slider", var = "minShownCooldown", default = 10, label = "Minimum Shown Cooldown", min = 2, max = 600, step = 1 },
		{ type = "Slider", var = "maxShownCooldown", default = 0, label = "Maximum Shown Cooldown", min = 0, max = 600, step = 1 },
	},
};

-- Variables
local plugin = AzCastBar.CreateMainBar("Frame","Cooldowns",extraOptions,true);
local timers = LibTableRecycler:New();

--------------------------------------------------------------------------------------------------------
--                                            Frame Scripts                                           --
--------------------------------------------------------------------------------------------------------

-- OnUpdate
local function OnUpdate(self,elapsed)
	local timer = self.timer;
	-- Progression
	if (not self.fadeTime) then
		self.timeLeft = (timer.endTime - GetTime());
		if (self.timeLeft < 0) then
			self.timeLeft = 0;
		end
		self.status:SetValue(self.timeLeft);
		self:SetTimeText(self.timeLeft);
		if (self.timeLeft == 0) then
			self.fadeTime = self.cfg.fadeTime;
		end
	-- FadeOut
	elseif (self.fadeElapsed <= self.fadeTime) then
		self.fadeElapsed = (self.fadeElapsed + elapsed);
		self:SetAlpha(self.cfg.alpha - self.fadeElapsed / self.fadeTime * self.cfg.alpha);
	else
		self:Hide();
		timers:RecycleIndex(self.index);
		plugin:QueryCooldowns();
	end
end

-- Cooldown Update
function plugin:SPELL_UPDATE_COOLDOWN(event)
	self:QueryCooldowns();
end

--------------------------------------------------------------------------------------------------------
--                                                Code                                                --
--------------------------------------------------------------------------------------------------------

local function SortCooldownsFunc(a,b)
	return a.endTime > b.endTime;
end

-- ConfigureBar
function plugin:ConfigureBar(bar)
	bar = (bar or self);
	bar:SetScript("OnUpdate",OnUpdate);
	return bar;
end

-- Query Cooldowns
function plugin:QueryCooldowns()
	timers:Recycle();
	for tab = 1, MAX_SKILLLINE_TABS do
		local name, texture, offset, numSpells, highestRankOffset, highestRankNumSpells = GetSpellTabInfo(tab);
		if (not name) then
			break;
		end
		for i = highestRankOffset + 1, highestRankOffset + highestRankNumSpells do
			i = GetKnownSlotFromHighestRankSlot(i);
			local start, duration, enabled = GetSpellCooldown(i,BOOKTYPE_SPELL);
			if (enabled ~= 0) and (duration) and (duration > self.cfg.minShownCooldown) and (self.cfg.maxShownCooldown == 0 or duration < self.cfg.maxShownCooldown) then
				local tbl = timers:Fetch();
				tbl.name = GetSpellName(i,BOOKTYPE_SPELL); tbl.duration = duration; tbl.startTime = start; tbl.endTime = start + duration; tbl.texture = GetSpellTexture(i,BOOKTYPE_SPELL);
			end
		end
	end
	sort(timers,SortCooldownsFunc);
	self:UpdateTimers();
end

-- Updates Timers
function plugin:UpdateTimers()
	for index, timer in ipairs(timers) do
		local bar = self.bars[index] or self:ConfigureBar(AzCastBar.CreateBar("Frame",self));

		bar.index = index;
		bar.timer = timer;
		bar.totalTimeText = (self.cfg.showTotalTime and " / "..FormatTime(timer.duration) or nil);

		bar.status:SetMinMaxValues(0,timer.duration);
		bar.status:SetStatusBarColor(unpack(self.cfg.colNormal));
		bar.icon:SetTexture(timer.texture);
		bar.name:SetText(timer.name);

		bar.fadeTime = nil;
		bar.fadeElapsed = 0;
		bar:SetAlpha(self.cfg.alpha);
		bar:Show();
	end
	-- Hide the rest
	for i = #timers + 1, #self.bars do
		self.bars[i]:Hide();
	end
end

-- OnConfigChanged
function plugin:OnConfigChanged(cfg)
	if (cfg.enabled) then
		self:RegisterEvent("SPELL_UPDATE_COOLDOWN");
		self:QueryCooldowns();
	else
		self:UnregisterAllEvents();
		timers:Recycle();
		self:UpdateTimers();
	end
end

--------------------------------------------------------------------------------------------------------
--                                          Initialise Plugin                                         --
--------------------------------------------------------------------------------------------------------

plugin:ConfigureBar();