-----------------------------------
-----------------------------------
-- SpamSentry by Anea
--
-- Detects and blocks gold-spam messages
-- Type /sentry or /spamsentry in-game for options
-----------------------------------
-- core.lua
-- Main routines and functionionality
-----------------------------------

-- Create Ace3 instance
SS = LibStub("AceAddon-3.0"):NewAddon("SpamSentry", "AceConsole-3.0", "AceHook-3.0", "AceEvent-3.0", "AceTimer-3.0")
local L = LibStub("AceLocale-3.0"):GetLocale("SpamSentry")
local wholib = LibStub:GetLibrary('LibWho-2.0')

-- Default settings and variables
local defaults = {
  profile = {
    modules = {},                      -- Module settings
    version = 0,                       -- The version we are running
    channelList = {                    -- Monitored channels
                   [L["whisper"]] = true,
                   [L["say"]] = true,
                   [L["yell"]] = true,
                   [L["general"]] = true,
                   [L["trade"]] = true,
                   [L["guildrecruitment"]] = true,
                   [L["emote"]] = true,
                   [L["lookingforgroup"]] = true,
                   [L["localdefense"]] = true,
                  },
    totalBlocked = 0,                  -- Blocked messages
    enableDelay = true,                -- Enables delaying of messages
    notifyMessage = true,              -- Shows a warning when a message has been blocked
    notifyHourly = true,               -- Shows an hourly reminder when messages have been blocked
    notifyDebug = false,               -- Shows debug messages
    minimumLevel = 1,                  -- Set the minimum level of players to be able to whisper you
    hidePartyInvite = false,           -- Hide party-invites from people you don't know
  },
  realm = {
  },
  char = {
    ignoreList = {},                   -- List with temporarily ignored players
  }
}

-- local variables
SS.currentBuild = 20091214           -- Latest build
SS.variablesLoaded = false           -- True once the mod has properly started up.

SS.spamReportList = {}               -- List with caught spammers
SS.message = {}                      -- Spam and Ham scores from messages
SS.character = {}                    -- List with characters
SS.characterBlackList = {}           -- Blacklisted characters for this session
SS.spamFeedbackList = {}             -- List with manually reported spammers cached for later feedback

SS.lastPlayer = ""                   -- Last player that a who-query was send for
SS.lastMessage = ""                  -- Last message that was send to you (skip duplicate entries sent by the interface)
SS.chatHistory = {}                  -- Chat cache
SS.chatQueue = {}                    -- Queue with messages that need a bit of delaying
SS.whoQueue = {}                     -- Queue with players we want to look up

SS.guildList={}                      -- Cache of the guildlist
SS.partyList={}                      -- Cache of the current party/raid
SS.friendsList = {}                  -- Cache of the friendslist
SS.knownList={}                      -- Cache of people you have talked to
SS.statistics={seen=0,delayed=0,who=0,spam=0}  -- For debugging purposes only

SS.strip = "[^abcdefghijklmnopqrstuvwxyz0123456789$\194\163\226\130\172=,.]+"    -- Pattern for stripping spacers from messages.
SS.clean = "[^a-zA-Z0-9$\194\163\226\130\172=,.%s]+"  -- Pattern for cleaning messages.

SS.blackList = {
                "$%d+.+%d+g",
                "%d+g.+$%d+",
                "%d+e.?u.?r.+%d+g",
                "%d+g.+%d+e.?u.?r",
                "%d+g.+e.?u.?r%d+",
                "\226\130\172%d+.+%d+g",
                "%d+g.+\226\130\172%d+",
                "\194\163%d+.+%d+g",
                "%d+g.+\194\163%d+",
                "freegold",
                "powerlevel","powerlveel",
                "1to70", "1to80",
                "pl170", "pl180",
                "gameworker",
                "wowgold",
                "buysomegold",
               }
SS.greyList = {
                "dollar", "pounds", "usd", "gbp",
                "www", "[,.]com", "[,.]corn", "[,.]conn", "[,.]c0m", "[,.]c0rn", "[,.]c0nn", "dotcom","cRT2m","[,.]cqm",
                "cheap",
                "buy", "kauf",
                "delivery",
                "discount", "rabatt",
                "peons",
                "170", "180",
                "375", "450",
                "gold",
                "%dkgold",
                "payment",
                "bucks",
                "safe",
                "statchanger",
                "hack",
                "20,000",
                "100,000",
                "eur",
                "code",
                "bonus",
                "stock",
                "company",
              }
SS.websites = {
                "100g.ca",
                "29games",
                "2joygame",
                "365ige",
                "4wowgold",
                "5uneed",
                "agamegold",
                "agamegoid",
                "ak774.com",
                "battleeurope",
                "battleadminworldofwarcraft",
                "benzgold",
                "blizzmounts",
                "blizzardusworldof",
                "brothergame",
                "buyeuwow",
                "cheapsgold",
                "cocwow",
                "coolwlk",
                "dewowgold",
                "dgamesky",
                "e4gold",
                "eugspa",
                "fastgg",
                "gagora",
                "gamenoble",
                "gmworker",
                "gmw0rker",
                "gmworking",
                "gmw0rking",
                "godmod",
                "gold4guild",
                "gold4shop",
                "goldwow",
                "goodgolds",
                "happygolds",
                "happyleveling",
                "helpugame",
                "heygt",
                "heypk",
                "igm365",
                "igs36five",
                "igs365",
                "gm963",
                "itembay",
                "itemrate",
                "iuc365",
                "k4gold",
                "loginusbattle",
                "love4gold",
                "m8gold",
                "marketgold",
                "mmoinn",
                "mmospa",
                "mountswow",
                "ogrm4",
                "pkpkg",
                "pvpbank.com",
                "scggame",
                "scswow",
                "ssegames",
                "svswow",
                "susanexpress",
                "www.-tbgold",
                "tbgold.-com",
                "tbowow",
                "tebuy",
                "terrarpg",
                "ucgogo",
                "usbattleworldofwarcraft",
                "usblizzard.net",
                "usblizzardworldof",
                "usbizzmounts",
                "vsvgame",
                "whoyo",
                "worldofgolds",
                "worldofwarcraftblizz",
                "wow4s",
                "wow7gold",
                "wowarmoryeu",
                "wowarmoryus",
                "wowcoming",
                "wowcnn",
                "wowdupe",
                "woweurope.cn",
                "wowforever",
                "wowgamebay",
                "wowgamelife",
                "wowgoldget",
                "wowgoldbuy",
                "wowgoldsky",
                "wowgoldex",
                "wowjx",
                "wowmine",
                "wowpanning",
                "wowpfs",
                "wowqueen",
                "wowseller",
                "wowspa",
                "wowsupplier",
                "wowtoolbox",
                "yesdaq",
                "zlywy",
              } 

-----------------------------------
-----------------------------------
-- Initialisation functions

function SS:OnInitialize()
  SS.db = LibStub("AceDB-3.0"):New("SpamSentryDB", defaults, "Default")
  SS:SetupOptions()
end

function SS:OnEnable()
  -- Cache guild and partylist
  SS:RegisterEvent("PLAYER_GUILD_UPDATE", SS.UpdateGuildList)
  SS:RegisterEvent("PARTY_MEMBERS_CHANGED",SS.UpdatePartyList)
  SS:RegisterEvent("FRIENDLIST_UPDATE", SS.UpdateFriendsList)
  
  SS:UpdateGuildList()
  SS:UpdatePartyList()
  SS:UpdateFriendsList()
  
  -- Register party-invite requests
  SS:RegisterEvent("PARTY_INVITE_REQUEST", SS.CheckPartyInvite)

  -- Register localised blacklist
  SS.localisedBlackList = L:LocalisedBlackList()
  
  -- Load Modules  
  for k, v in self:IterateModules() do
    if self.db.profile.modules[k] ~= false then
      v:Enable()
    end
  end
  
  -- Schedule hooks to be set as soon as the game has fully loaded
  SS:RegisterEvent("PLAYER_ENTERING_WORLD", function() SS:ScheduleTimer(SS.StartUp, 5) end)
end

function SS:OnDisable()
  SS:FlushChatQueue()           -- Show all delayed messages  now
  SS:UnhookAll()                -- Disable all hooks
  SS.variablesLoaded = false    -- Flag this mod as disabled
  SS:CancelAllTimers()
  
  -- UnLoad Modules  
  for k, v in self:IterateModules() do
    if self.db.profile.modules[k] ~= false then
      v:Disable()
    end
  end  
end

function SS:StartUp()
  if not SS.variablesLoaded then
    SS:UnregisterEvent("PLAYER_ENTERING_WORLD")
    SS:RawHook("ChatFrame_MessageEventHandler", SS.ChatFrame_MessageEventHandler, true)   -- Hook the Chatframe OnEvent function.
    SS:RawHook("SetItemRef", "SetItemRef", true)                                          -- Hook the ItemLink OnEvent function
    SS:SecureHook("UnitPopup_OnClick")                                                    -- Hook the UnitPopup OnClick function
    SS:Compat_Enable()                                                                    -- Hook 3rd party addons

    -- Clear the ignorelist from last session
    SS:ClearIgnore(0)
 
    -- Check version, update variables
    SS:CheckVersion()
    
    -- Start queue-system
    SS:ToggleChatQueue()
    
    -- Schedules
    SS:ScheduleRepeatingTimer(SS.CheckReport, 3600)     -- Hourly report message
    SS:ScheduleRepeatingTimer(SS.ChatQueueCooldown, 5)  -- Who-callback check
    SS:ScheduleRepeatingTimer(SS.CollectGarbage, 60)    -- Garbage collection
        
    SS.variablesLoaded = true
    SS:CheckReport()
    SS:Msg(3, "SpamSentry loaded")
  end
end

-----------------------------------
-----------------------------------
-- Hooks and events

-- Chat message event
function SS:ChatFrame_MessageEventHandler(event, handler)
  local cf = self
  local msg = arg1
  local plr = arg2
  local chn = nil
  local source = strlower(tostring(arg4))
  local id = arg11
  local spam = -2
  local channels = SS.db.profile.channelList
  local tmsg = ""
  
  if msg and plr then
    if event == "CHAT_MSG_WHISPER" and channels[L["whisper"]] then
      SS.knownList[plr] = true
      chn = L["whisper"]
    elseif event== "CHAT_MSG_SAY" and channels[L["say"]] then
      chn = L["say"]
    elseif event== "CHAT_MSG_YELL" and channels[L["yell"]] then
      chn = L["yell"]
    elseif event=="CHAT_MSG_EMOTE" and channels[L["emote"]] then
      chn = L["emote"]
    elseif event=="CHAT_MSG_CHANNEL" then
      if channels[L["trade"]] and strfind(source, L["trade"]) then
        chn = L["trade"]
      elseif channels[L["general"]] and strfind(source, L["general"]) then
        chn = L["general"]
      elseif channels[L["guildrecruitment"]] and strfind(source, L["guildrecruitment"]) then
        chn = L["guildrecruitment"]
      elseif channels[L["lookingforgroup"]] and strfind(source, L["lookingforgroup"]) then
        chn = L["lookingforgroup"]
      elseif channels[L["localdefense"]] and strfind(source, L["localdefense"]) then
        chn = L["localdefense"]
      end
    elseif (strfind(event, "CHAT_MSG_SYSTEM") and SS:SupressIgnoreMsg(msg)) then
      return
    end
  end
  if chn then
    spam = SS:SpamCheck1(msg, plr, chn, event)
  end
  
  SS.statistics.seen = SS.statistics.seen + 1
  if spam == -2 then
    SS:CallChatEvent(self, event, handler)
  elseif spam == -1 then
    if SS.db.profile.enableDelay then
      local mIndex = SS:AddChatQueue(plr, msg, event, chn, id, handler, self)
      SS.chatQueue[mIndex].queued = true
      SS.statistics.delayed = SS.statistics.delayed + 1
    else
      SS:CallChatEvent(self, event, handler)
    end
  elseif spam == 1 then
    SS:SpamFound(plr, SS:GetChatHistory(plr), chn, id)
    SS.knownList[plr] = false
  elseif spam ==2 then
    SS.knownList[plr] = false
  elseif spam == 0 then
    SS:SendWho(plr, msg, event, chn, id, handler, self)
  end
end

-- Set query for player info
function SS:SendWho(plr, msg, event, chn, id, handler, cf)
  local result = wholib:UserInfo(plr, 
                               {
                                 queue = SS.WHOLIB_QUEUE_QUIET, 
                                 timeout = -1,
                                 callback = "WhoCallback",
                                 handler = SS,
                               })
  local index = SS:AddChatQueue(plr, msg, event, chn, id, handler, cf)
  if result then
    SS.character[result.Name] = {}
    SS.character[result.Name].level = result.Level or 1
    SS.character[result.Name].guild = result.Guild or ""
    SS:ChatQueueProcessMessage(index)
  else
    SS.statistics.who = SS.statistics.who + 1
    SS.chatQueue[index].waiting = true
  end
end

-- Callback function for who-query results
function SS:WhoCallback(result)
  if not result then return end
  
  SS.character[result.Name] = {}
  SS.character[result.Name].level = result.Level or 1
  SS.character[result.Name].guild = result.Guild or ""
  for i=1, #SS.chatQueue, 1 do
    if SS.chatQueue[i].name == result.Name then
      SS:ChatQueueProcessMessage(i)
    end
  end
end

-- This function hooks into clickable chatlinks. It does some action if special SpamSentry links are found.
function SS:SetItemRef(link, text, button)
  if strfind(link, "SpamSentrySpamTicket") then
    local report = SS:GetModule('Report', true)
    if report then
      report:ShowGUI()
    else
      SS:Msg(0, "Report module not available.")
    end
  elseif strfind(link, "SpamSentryMsg") then
    local s,e,entry = string.find(link, "(%d+)")
    entry = tonumber(entry)
    if SS.spamReportList[entry] then
      local plr = SS.spamReportList[entry].player
      SS:Msg(0, "* "..SS:PlayerLink(plr)..": |cffff55ff"..SS.spamReportList[entry].message.."|r")
    end
  else
    SS.hooks["SetItemRef"](link, text, button)
  end
end

-- Hook to unit-pop-up menu's
function SS:UnitPopup_OnClick(self)
  local dropdownFrame = UIDROPDOWNMENU_INIT_MENU;
  local button = self.value;
  local unit = dropdownFrame.unit;
  local name = dropdownFrame.name;
  -- Report spammer
  if button == "REPORT_SPAM" then
    if (unit and not name) then name = UnitName(unit) end
    if name then
      local msg = SS:GetChatHistory(name)
      if msg and (msg ~= "") then
        SS.spamFeedbackList[name] = msg
        SS:Msg(0, format(L["%s has been added to the SpamSentry feedback list. CTRL-click the SpamSentry icon to open the feedback panel."], name))
      end
    end
  end
end

-- Outgoing chatevents
function SS:CallChatEvent(cf, event, handler)
  if handler and type(handler)=="function" then
    handler(event, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12)
  else
    SS.hooks["ChatFrame_MessageEventHandler"](cf, event, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12)
  end
end

-- Called when a chat-event needs to be thrown while being in a who-event.
function SS:CallOldChatEvent(index)
  if SS.chatQueue[index].done == false then
    SS.chatQueue[index].done = true
    local sthis, sevent, sarg1, sarg2, sarg3, sarg4, sarg5, sarg6, sarg7, sarg8, sarg9, sarg10, sarg11, sarg12 = this, event, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12
    local m = SS.chatQueue[index]
    local cf
    this, event, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12, handler, cf = m.this, m.event, m.arg1, m.arg2, m.arg3, m.arg4, m.arg5, m.arg6, m.arg7, m.arg8, m.arg9, m.arg10, m.arg11, m.arg12, m.handler, m.chatframe
    SS:CallChatEvent(cf, event, handler)
    this, event, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12 = sthis, sevent, sarg1, sarg2, sarg3, sarg4, sarg5, sarg6, sarg7, sarg8, sarg9, sarg10, sarg11, sarg12
  end 
  
end

-----------------------------------
-----------------------------------
-- Spam detection

-- This function checks the message for spam. Scores are assigned for various characteristics
-- Returns 2 on blacklist, 1 on spam, 0 on yet unknown, -1 on no spam and queue, -2 whitelist
-- Returns the concatenated message if applicable
function SS:SpamCheck1(msg, plr, chn, evt)
  -- Check messaging and garbage collection
  SS:ChatQueueCooldown()
  
  -- You self are whitelisted!
  if strsub(evt, 10) == "WHISPER_INFORM" then return -2 end
  if plr==UnitName("player") and not SS.db.profile.notifyDebug then return -2 end   -- Extra check allows debugging

  -- Player on Character blacklist has spammed before and is ignored now
  if SS:InList(SS.characterBlackList, plr) then return 2 end

  -- Player in your party/raid is white-listed
  if SS:InList(SS.partyList, plr) then return -2 end

  -- Player on your friends-list is white-listed
  if SS:InList(SS.friendsList, plr) then return -2 end
  
  -- Player in your guild is white-listed
  if SS:InList(SS.guildList, plr) then return -2,msg end
  
  -- GM is whitelisted
  if(TEXT(getglobal("CHAT_FLAG_"..tostring(arg6)) == CHAT_FLAG_GM)) then return -2,msg end
  
  -- Previous messages from players within the past 60 seconds are taken into account.
  SS:AddChatHistory(plr, msg, chn)
  msg = SS:GetChatHistory(plr)
   
  -- Blacklist items score -0.1 point for the first one, -0.2 for the next, -0.4 for the one after, etc.
  local sc,gr = SS:ScoreSpam(plr, msg)
  
  if sc >= 0 and not ((SS.db.profile.minimumLevel > 1) and (chn == L["whisper"])) then 
    -- If no negative score has been found, there's no reason to do more checks or do a who-query.
    -- This should prevent Bank / Auction and other windows from closing most of the time.
    if gr == 0 then
      return -2
    end
    return -1
  elseif SS.character[plr] then
    -- Check if the player has been queried before. If so, no need to do a who-query again.
    -- This lowers server-load and prevents messages passing through when spammed fast after each other
    local tsc = sc + SS:ScoreHam(plr, msg)
    if tsc < 0 then
      return 1
    elseif SS.character[plr].level < SS.db.profile.minimumLevel then
      return 2
    elseif sc == 0 and gr ==0 then 
      -- Whitelist if no suspicious items have been found
      return -2
    end
    return -1
  end
  -- Message is suspicious, but we have no data on this player. Who-query advised.
  return 0
end

-- Secundairy spamcheck afer who-request has been completed or timed-out
-- Returns 2 on blacklist, 1 on spam, 0 on yet unknown, -1 on no spam and queue, -2 whitelist
function SS:SpamCheck2(plr, msg)
  local s,g = SS:ScoreSpam(plr, msg)
  if SS.character[plr] and SS.character[plr].level < SS.db.profile.minimumLevel then
    return 2
  elseif SS:ScoreHam(plr, msg) + s < 0 then
    return 1
  else
    if SS.db.profile.enableDelay then
      return -1
    else
      return -2
    end
  end
end

-- Looks up items from the list in the message and assigns scores
function SS:ScoreSpam(plr, msg)
  if SS.message[msg] and SS.message[msg].spam then
    return SS.message[msg].spam, SS.message[msg].grey
  end
  local score,grey,feedback = 0,0,{}
  -- Remove all spacers and non-latin characters
  local tmsg = string.gsub(strlower(msg), SS.strip, "")
  -- Parse websites
  for i=1, #SS.websites do
    if strfind(tmsg, strlower(SS.websites[i])) then
      score = score==0 and -0.4 or score * 2
      tinsert(feedback, SS.websites[i])
    end
  end
  -- Parse blacklist
  for i=0, #SS.blackList, 1 do
    if SS.blackList[i] and strfind(tmsg, strlower(SS.blackList[i])) then
      score = score==0 and -0.1 or score * 2
      tinsert(feedback, SS.blackList[i])
    end
  end
  -- Parse localised blacklist
  if SS.localisedBlackList then
    for i=0, #SS.localisedBlackList, 1 do
      if SS.localisedBlackList[i] and strfind(msg, strlower(SS.localisedBlackList[i])) then 
        score = score==0 and -0.2 or score * 2
        tinsert(feedback, SS.localisedBlackList[i])
      end
    end
  end
  -- Parse greylist. If a greylisted item is found, the message will be queued for delay. 
  -- If blacklisted items were found above, the grey score will be added to the spam score
  for i=0, #SS.greyList, 1 do
    if SS.greyList[i] and strfind(tmsg, strlower(SS.greyList[i])) then 
      grey = grey==0 and -0.1 or grey * 2
      tinsert(feedback, SS.greyList[i])
    end
  end
  
  -- Take grey score into consideration; add it to spamscore if we found blacklisted items or 3 or more grey listed items.
  if score < 0 then 
    score = score + grey
  elseif grey < -0.2 then 
    score = score + grey 
  end

  -- If the message contains a link to an item, add 0.1 points to the score.
  -- Note that this is actually hamscoring, but I have added it here to prevent from unneeded who-queries
  if strfind(msg, "Hitem[0-9]+h") then
    score = score + 0.1
  end

  -- Cache the scores so that we don't have to recalculate them again for this message.
  if not SS.message[msg] then
    SS.message[msg] = {}
  end
  SS.message[msg].spam = score
  SS.message[msg].grey = grey
  SS.message[msg].time = GetTime()
  
  return score,grey,feedback
end

function SS:ScoreHam(plr, msg)
  local score = 0

  if SS.character[plr] then
    local level = SS.character[plr].level
    local guild = SS.character[plr].guild

    if level and level>10 then
      score = score + 0.1  -- Refund if the player is above level 10
    end
    if level and level>20 then
      score = score + 0.3  -- Refund more if the player is above level 20
    end
    if guild and guild~= "" then
      score = score + 0.3  -- Refund if the player is in a guild
    end
  end

  -- If your own level is below 15, add 0.2 points to the score, as you are more likely to get messages from low-levels
  if UnitLevel("player")<15 then
    score = score + 0.2
  end
  return score
end

-----------------------------------
-----------------------------------
-- Report functions
-- Called when a spammer is found

function SS:SpamFound(plr, msg, typ, id)
  if SS:InList(SS.characterBlackList, plr) then return end
  SS.statistics.spam = SS.statistics.spam + 1
  local entry = SS:AddSpam(plr, msg, typ, id)
  if SS.db.profile.notifyMessage then
    local link = "|HSpamSentrySpamTicket|h|cff8888ff["..L["here"].."]|r|h"
    local text = format(L["* Alert: %s tried to send you %s (%s). Click %s to report."], SS:PlayerLink(plr), SS:MessageLink(L["this message"],entry), typ, link)
    SS:Msg(2, text)
    PlaySound("QUESTLOGOPEN")
  end
end

-- Adds the player to the reportlist.
-- An event is fired if and only if the respective player has not been marked as spammer before:
-- Event: SPAMSENTRY_SPAM_FOUND
-- Arguments: Player, Message, Channel, Message_ID
function SS:AddSpam(plr, msg, typ, id)
  SS:AddIgnore(plr, typ)
  SS.db.profile.totalBlocked = SS.db.profile.totalBlocked + 1
  if not SS:InList(SS.characterBlackList, plr) then
    tinsert(SS.characterBlackList, plr)
  end
  local entry = SS.InList(SS.spamReportList, plr)
  if not entry then
    -- clean-up message
    local tmsg = gsub(tostring(msg), SS.clean, "")
    tmsg = gsub(msg, "([%s=.,])+", "%1")
    local datetime = date("%x - %H:%M:%S", time())
    local sum = datetime.."\n"..tmsg
    tinsert(SS.spamReportList, { player = plr,
                                   message = tmsg,
                                   time = datetime,
                                   type = typ or "unknown",
                                   id = id,
                                   summary = sum,
                                 })
    SS:SendMessage("SPAMSENTRY_SPAM_FOUND", plr, msg, typ, id)
    return #SS.spamReportList
  end
  return entry
end

-- Show a warning once every hour.
function SS:CheckReport()
  local m = L["One or more characters are on the reportlist. Click %s to report them to a GM."]
  -- Create a clickable link. This is handled by the SetItemRef hook.
  if #SS.spamReportList>0 then
    local link = "|HSpamSentrySpamTicket|h|cff8888ff["..L["here"].."]|r|h"
    SS:Msg(1, format(m, link))
    PlaySound("QUESTLOGOPEN")
  end
end

-----------------------------------
-----------------------------------
-- Queue functions

function SS:AddChatQueue(plr, msg, e, chn, id, hnd, cf)
  tinsert(SS.chatQueue,{name = plr,
                          message = msg,
                          time = GetTime(),
                          waiting = false,
                          queued = false,
                          done = false,
                          id = id,
                          channel = chn,
                          this = this,
                          event = e,
                          arg1 = arg1,
                          arg2 = arg2,
                          arg3 = arg3,
                          arg4 = arg4,
                          arg5 = arg5,
                          arg6 = arg6,
                          arg7 = arg7,
                          arg8 = arg8,
                          arg9 = arg9,
                          arg10= arg10,
                          arg11= arg11,
                          arg12 = arg12,
                          handler = hnd,
                          chatframe = cf,
                        })
  return #SS.chatQueue
end

function SS:CheckChatQueue()
  if #SS.chatQueue > 0 then
    local t = GetTime() - 5
    local i = 1
    local stop = false
    repeat
      if SS.chatQueue[i].queued and (SS.chatQueue[i].time < t) and not SS.chatQueue[i].done then
        local plr = SS.chatQueue[i].name
        if not SS:InList(SS.characterBlackList, plr) then  -- Check blacklist.
          SS:CallOldChatEvent(i)
        end
        tremove(SS.chatQueue, i)
      else
        i = i + 1  
      end
      stop = i >= #SS.chatQueue
    until stop
  end
end

function SS:FlushChatQueue()
  local num = #SS.chatQueue
  for i=1, num, 1 do
    local plr = SS.chatQueue[1].args[4]
    if not SS:InList(SS.characterBlackList, plr) then  -- Check blacklist.
      SS:CallOldChatEvent(1)
    end
    tremove(SS.chatQueue, 1)
  end
end

function SS:ToggleChatQueue()
  if SS.db.profile.enableDelay then
    SS:ScheduleRepeatingTimer(SS.CheckChatQueue, 1)
  else
    SS:CancelTimer("CheckChatQueue", true)
    SS:FlushChatQueue()
  end
end

local waiting, timesup, queued, done
-- Removes players from waitlist if they've been on it for more then 10 seconds
function SS:ChatQueueCooldown()
  local t = GetTime() - 10
  for i=1, #SS.chatQueue, 1 do
    waiting = SS.chatQueue[i].waiting
    timesup = SS.chatQueue[i].time < t
    queued = SS.chatQueue[i].queued
    done = SS.chatQueue[i].done
    if waiting and timesup and not (queued or done) then
      SS:ChatQueueProcessMessage(i)
    elseif not (waiting or queued or done) then
      -- Mark left-behind messages as garbage (just-in-case)
      SS.chatQueue[i].done = true
    end
  end
end

-- Process a message after a who result has been received or timed out
function SS:ChatQueueProcessMessage(index)
  local name = SS.chatQueue[index].name
  local msg = SS:GetChatHistory(name)
  local spam = SS:SpamCheck2(name, msg)
  if spam == 2 then
    SS.chatQueue[index].done = true
    SS.knownList[name] = false
  elseif spam == 1 then
    SS:SpamFound(name, msg, SS.chatQueue[index].channel, SS.chatQueue[index].id)
    SS.chatQueue[index].done = true
    SS.knownList[name] = false
  elseif spam == -1 then
    SS.chatQueue[index].queued = true
  elseif spam == -2 then
    SS:CallOldChatEvent(index)
  end
end

-- Garbage collection of chatqueue and message cache
function SS:CollectGarbage()
  -- Message cache
  if #SS.chatQueue > 0 then
    local i = 1
    local stop = false
    repeat
      if SS.chatQueue[i].done then
        tremove(SS.chatQueue, i)
      else
        i = i + 1 
      end
      stop = i>= #SS.chatQueue
    until stop
  end

  -- Spam / Ham cache
  if #SS.message > 0 then
    local i = 1
    local t = GetTime() - 120
    repeat
      if SS.message[i].time < t then
        tremove(SS.message, i)
      else
        i = i + 1
      end
      stop = i>= #SS.message
    until stop
  end

  -- ChatHistory: The last message is saved for 1 minute, additional messages for 30 seconds
  local t = GetTime()
  for i=1, #SS.chatHistory, 1 do
  -- Remove first message if older then 1 minutes
  if SS.chatHistory[i].time[1] < t - 60 then
      tremove(SS.chatHistory,i)
    else
    -- Remove all other messages if older then 30 seconds
      for j=2, #SS.chatHistory[i].time, 1 do
        if SS.chatHistory[i].time[j] < t - 30 then
          tremove(SS.chatHistory[i].message,j)
          tremove(SS.chatHistory[i].time,j)
          tremove(SS.chatHistory[i].channel,j)
        end
      end
    end
  end
end

-----------------------------------
-----------------------------------
-- Cache and history functions

function SS:UpdateGuildList()
  SS.guildList = {}
  if GetGuildInfo("player") then
    for i=1,GetNumGuildMembers(true),1 do
      local name = GetGuildRosterInfo(i)
      tinsert(SS.guildList, name)
    end
  end
end

function SS:UpdatePartyList()
  SS.partyList = {}
  if UnitInRaid("player") then
    for i=1,GetNumRaidMembers(),1 do
      local name = GetRaidRosterInfo(i)
      tinsert(SS.partyList, name)
    end
  else
    for i=1,GetNumPartyMembers(),1 do
      local name = UnitName("party"..tostring(i))
      tinsert(SS.partyList, name)
    end
  end
end

function SS:UpdateFriendsList()
  SS.friendsList = {}
  local numfriends = GetNumFriends()
  for i=1, numfriends, 1 do
    local name = GetFriendInfo(i)
    tinsert(SS.friendsList, name)
  end
end

-- The chathistory keeps track of recently received messages from a player
function SS:AddChatHistory(plr, msg, chn)
  -- Insert new message
  if not SS.chatHistory[plr] then
    SS.chatHistory[plr] = { message = {}, time = {}, channel = {}}
  end
  if not (SS.chatHistory[plr].message[1] == msg) then
    tinsert(SS.chatHistory[plr].message, 1, msg)
    tinsert(SS.chatHistory[plr].time, 1, t)
    tinsert(SS.chatHistory[plr].channel, 1, chn)
  end
end

-- Returns the combined recent messages from this player
function SS:GetChatHistory(plr)
  local msg = ""
  if SS.chatHistory[plr] then
    for i=1, #SS.chatHistory[plr].message, 1 do
      msg = SS.chatHistory[plr].message[i].." "..msg
    end
  end
  return msg
end

-----------------------------------
-----------------------------------
-- Ignore party invites from strangers
function SS:CheckPartyInvite(plr)
  if not SS.db.profile.hidePartyInvite then return end
  if not (SS:InList(SS.friendsList, plr) or SS:InList(SS.guildList, plr) or SS.knownList[plr]) then
    for i=1, STATICPOPUP_NUMDIALOGS do
      local f = getglobal("StaticPopup" .. i)
      if f:IsVisible() and f.which=="PARTY_INVITE" then 
        f:Hide()
        SS:Msg(0, L["Player unknown, party invite cancelled"]..": "..SS:PlayerLink(tostring(plr)))
      end
    end
  end
end

-----------------------------------
-----------------------------------
-- Ignore functions
-- These function assure that subsequent textballoons from spammers are suppressed
function SS:AddIgnore(plr, typ)
  if typ == L["say"] or typ == L["yell"] then
    if not SS:InList(SS.db.char.ignoreList, plr) then
      tinsert(SS.db.char.ignoreList, plr)
      AddIgnore(plr)
    end
  end
end

function SS:ClearIgnore(num)
  if num==0 then
    num = #SS.db.char.ignoreList
  end
  for i=1,num,1 do
    local plr = SS.db.char.ignoreList[1]
    if plr then
      DelIgnore(plr)
    end
    SS:ScheduleTimer(function() tremove(SS.db.char.ignoreList,1) end, 2)
  end
end

function SS:SupressIgnoreMsg(msg)
  if strfind(msg, ERR_IGNORE_NOT_FOUND) or strfind(msg, ERR_IGNORE_SELF) or strfind(msg, ERR_IGNORE_DELETED) then return true end
  for i=1, #SS.db.char.ignoreList, 1 do
    if strfind(msg, SS.db.char.ignoreList[i]) then
      return true
    end
  end
  return false
end

-----------------------------------
-----------------------------------
-- Utility and other functions

-- Output message to the chatframe
-- Level can be either:
-- 0: System message
-- 1: Hourly notification
-- 2: Normal message
-- 3: Debug
function SS:Msg(level, text)
  if not text then return end
  local s = SS.db.profile
  if level==0
  or level==1 and s.notifyHourly 
  or level==2 and s.notifyMessage
  or level==3 and s.notifyDebug then
    DEFAULT_CHAT_FRAME:AddMessage(text, 1, 0.6, 0.46)
  end
end

-- Returns the index of the item in the list, true/false for recursive lists
function SS:InList(list, item)
  if list and item then
    item = strlower(item)
    for i,v in pairs(list) do
      if type(v)=="table" then
        if SS:InList(v, item) then
          return i
        end
      elseif type(v)=="string" and strlower(v)==item then 
        return i
      elseif v==item then
        return i
      end
    end
  end
  return false
end

-- Create a link to a playername
function SS:PlayerLink(link)
  local type = "player:"..link
  return "|H"..type.."|h|cffffff00["..link.."]|r|h"
end

-- Create a link to a SpamSentry message
function SS:MessageLink(link, entry)
  local type= "SpamSentryMsg:"
  return "|H"..type.."_"..entry.."|h|cff8888ff["..link.."]|r|h"
end

-- Show a dialog box with given text. Executes the given functions on accept/cancel.
function SS:ShowNotification(text, acceptText, acceptFunc, cancelText, cancelFunc)
  StaticPopupDialogs["SPAMSENTRY_NOTIFICATION"] = {
   text = text,
   button1 = acceptText,
   button2 = (type(cancelFunc)=="function") and cancelText or nil,
   OnAccept = acceptFunc,
   OnCancel = cancelFunc,
   timeout = 0,
   whileDead = 1,
   hideOnEscape = 1
  };
  StaticPopup_Show ("SPAMSENTRY_NOTIFICATION");
end

-- Functionality for versioning
function SS:CheckVersion()
--  if SS.db.profile.version < 20090119 then
--    SS:ShowNotification(
--      format("|cffff0000SpamSentry|r|cffff9977 v%s|r|cffffffff\n\n  \n\nAnea",20090119),
--      TEXT(OKAY),
--      function() end
--    )
--  end
  if not SS.db.profile.version == SS.currentBuild then
    SS:Msg(0, format(L["SpamSentry v%s by Anea. Type |cffffffff/sentry|r or right-click the icon for options."], "|cffffffff"..SS.db.profile.version.."|r"))
  end
  SS.db.profile.version = SS.currentBuild
end

-- Debug messages
function SS:Test(msg)
  SS:Msg(1, "Checking: '"..string.gsub(strlower(msg), SS.strip, "").."'")
  local s,g,f = SS:ScoreSpam("test", msg)
  local h = SS:ScoreHam("test", msg)
  local c = SS:SpamCheck2("test", msg)
  local class = {[-2]="Whitelisted", [-1]="Clean", [0]="Suspicious", [1]="Spam", [2]="Blacklisted"}
  local feedback = ""
  for i,v in pairs(f) do
    feedback = feedback.."\n  "..v
  end
  SS:Msg(1, "Spam: "..s..", Grey: "..g..", Ham: "..h..", Class: "..class[c])
  SS:Msg(1, "Feedback: "..feedback)
end

-- Print out feedback text
local hex2str = {["0"]=0,["1"]=1,["2"]=2,["3"]=3,["4"]=4,["5"]=5,["6"]=6,["7"]=7,["8"]=8,["9"]=9,["A"]=10,["B"]=11,["C"]=12,["D"]=13,["E"]=14,["F"]=15}
function SS:Feedback(s)
  local ret, a, b = "", 0, 0
  for i=1, strlen(s), 2 do
    a = hex2str[strsub(s,i,i)]
    b = hex2str[strsub(s,i+1,i+1)]
    if a and b then
      ret = ret..strchar((16 * a) + b)
    end
  end
  SpamSentry_TicketEditBox:SetText(ret)
  SpamSentryUITicketParent:Show()
end
